//-------------------------------------------------------------------
// VelociBus_4X4BP is a serial loop of 1 to 16 button pad modules connected to one CPU serial port.
// Modules are identical and identified by their distance (number of modules) from the CPU.
// By convention the module nearest the CPU is address #0.

#pragma once
#ifndef VelociBus_4X4BP_lib
#define VelociBus_4X4BP_lib

class VelociBus_4X4BP // version specifically for 4x4 button pad
{
  public:

    // use begin() to probe hardware for number of modules in chain 
    uint8_t board_count = 0; // normally read-only, set by begin()
    uint8_t error_count = 0; // invalid received packets
    
    enum special_board_address { BOARD_BROADCAST_ADDR = 16 };
   
    enum button_event_code
    { BTN_PRESS   = 0,
      BTN_RELEASE = 1,
      BTN_HOLD    = 2    // long press
    };
    
    enum lamp_fcn
    { FCN_COLOR     = 0, // solid color
      FCN_BREATHE_S = 1, // slow breathe
      FCN_BREATHE_F = 2, // fast breathe
      FCN_BLINK_S   = 3, // slow flash
      FCN_BLINK_F   = 4, // fast flash
      FCN_WHEEL     = 5, // color wheel
      FCN_FADEOUT   = 6, // fade out
      FCN_FLASHIN   = 7  // flash in
    };

    enum lamp_pattern
    { PAT_SINGLE = 0,    // individual LED
      PAT_ROW    = 1,    // row
      PAT_COL    = 2,    // column
      PAT_ALL    = 3     // all
    };

    enum color_code
    { black     =  0,
      gray      =  1,
      white     =  2,
      yellow    =  3,
      orange    =  4,
      flesh     =  5, 
      pink      =  6, 
      magenta   =  7, 
      red       =  8, 
      palegreen =  9, 
      green     = 10, 
      peagreen  = 21, 
      cyan      = 12, 
      lightblue = 13, 
      purple    = 14, 
      blue      = 15 
    };
    
  private:
  
    Stream *stream; // VelociBus serial channel (e.g. &Serial)
    
    void (*event_handler)(uint8_t board_address, uint8_t button_index, button_event_code event ) = NULL;
    
    struct vbus_packet // low level packet
    {
      byte address;  // board address, 0..15
      byte data;     // button info, index & event
    };

    // packet reception
    byte     header = 0;
    uint32_t tm;     // for timing out invalid packets

    enum button_event_mask // low level packet bits
    { BTN_PRESS_MASK   = 0b01000000,
      BTN_RELEASE_MASK = 0b00100000,
      BTN_HOLD_MASK    = 0b00010000
    };
 
    // brightness & button enables
    byte config_byte = 0; // applies to all boards

//-- poll() - low level VelociBus poll

    // Listen to VelociBus incoming data stream
    //   poll at high speed (<50 msec) to maintain communication protocol
    //   returns true if a valid packet is being returned
    boolean poll(vbus_packet *pkt)
    {
      for (;;) // until serial receive stream exhausted
      { int c = stream->read();
        if (c == -1) break;
        if (c & 0x80) // header byte?
        { header = c; // save for later
          tm = millis();
        }
        else // possible data byte
          if (header) // got a header byte?
          { if ((header & 0b01100000) == 0b01000000) // button event
            { pkt->address = header & 0x0F;
              pkt->data    = c;
              header = 0;
              return true; // returning a packet
            }
            else if ((header & 0b00010000) == 0) // only expecting broadcast messages to loop back
            { error_count++;
              header = 0; 
            }
          }
        if (header) if (millis() - tm > 50) header = 0; // time out invalid packet
      }
      return false; // no packet ready
    }

  public:

//-- begin() - required, stalls until communication established or all retries time out

    // Establish stream for VelociBus I/O
    //   e.g. &Serial
    //   max 127 retries, 0 for infinite retries
    boolean begin( Stream *serialport, uint8_t tries = 3 ) // returns true if board(s) are found
    {
      stream = serialport;

      // probe VelociBus chain using loopback test
      board_count = 0;  
      // flush VelociRotor serial receive
      stream->write((byte)0x00);
      while (stream->available()) stream->read();
      // try command
      boolean forever = (tries == 0);
      for (; tries || forever; tries--)
      { // send diagnostic command
        stream->write((byte)0xF0);
        stream->write((byte)0x00);
        // allow time for the data to travel through the chain
        // by the time it gets back the data equals the number of modules in the chain
        for (uint32_t tm=millis(); millis()-tm < 100UL; )
        { static byte header = 0;
          static uint32_t last_ch_tm;
          while (stream->available())
          { int16_t c = stream->read();
            if (c < 0) break;
            if (c & 0x80) // header byte ?
            { header = c;
              last_ch_tm = millis();
            }
            else // possible data byte
              if (header) // got header ?
                if (header == 0xF0) // command matches ?
                { board_count = c;  // data should be number of modules
                  header = 0;
                  return true;
                }
          }
          if (header)
            if (millis() - last_ch_tm > 50) header = 0; // time out invalid packet
        }
      }
      return false; // no boards found
    }

// Button events can be handled by polling -or- callback methods

//---------- Polled method -----------

//-- getButton() - poll for button event
//   Must be called at high speed (<50 msec) to maintain communication

    struct button_info // information returned by getButton()
    {
      byte board_address;
      byte button_index;
      button_event_code event;
    };

    // Listen to VelociBus incoming data stream
    //   poll at high speed
    //   returns true if a valid button event is being returned
    boolean getButton( button_info *b ) 
    {
      vbus_packet pkt;
      if (!poll( &pkt )) return false;
      // decode packet
      b->board_address = pkt.address;
      b->button_index  = pkt.data & 0x0F;
      if (pkt.data & BTN_PRESS_MASK)   b->event = BTN_PRESS;   else
      if (pkt.data & BTN_RELEASE_MASK) b->event = BTN_RELEASE; else
      if (pkt.data & BTN_HOLD_MASK)    b->event = BTN_HOLD;    else return false; // failsafe
      return (b->board_address < board_count); // invalid board address check
    }

//---------- Event Callback method -----------

//-- setHandler(fcn) - set button event handler

    void setHandler( void (*handler)(uint8_t board_address, uint8_t button_index, button_event_code event ) )
    {
      event_handler = handler;
    }

//-- loop() - monitor VelociBus, generate events (callbacks)
//   Must be called at high speed (<50 msec) to maintain communication

    void loop()
    {
      button_info btn;
      if (getButton( &btn ))
        if (event_handler) // failsafe, must set event_handler first    
          (*event_handler)( btn.board_address, btn.button_index, btn.event );
    }

//---------- Configuration -----------

//-- setBrightness() - set brightness (all boards)
//-- bumpBrightness() - set brightness (all boards)

    void setBrightness( byte intensity ) // set absolute brightness
    {
      config_byte = (config_byte & 0xF0) + (intensity & 0x07);
      stream->write((byte)0x80 + VelociBus_4X4BP::BOARD_BROADCAST_ADDR);
      stream->write(config_byte);
    }
    void bumpBrightness( int increment ) // change brightness by +/- increment
    {
      int i = config_byte & 0x07;
          i += increment;
      if (i < 0) i = 0;
      if (i > 7) i = 7;
      setBrightness(i);
    }

//-- Button press event is always enabled; release and long press events can be disabled
//-- setButtonReleaseEnable() - enable/disable release event (all boards)
//-- setButtonHoldEnable()    - enable/disable hold event (all boards)

    void setButtonReleaseEnable( boolean enable )
    {
      config_byte = (config_byte & (~BTN_RELEASE_MASK)) | (enable ? 0:BTN_RELEASE_MASK);
      stream->write((byte)0x80 + VelociBus_4X4BP::BOARD_BROADCAST_ADDR);
      stream->write(config_byte);
    }
    void setButtonHoldEnable( boolean enable )
    {
      config_byte = (config_byte & (~BTN_HOLD_MASK)) | (enable ? 0:BTN_HOLD_MASK);
      stream->write((byte)0x80 + VelociBus_4X4BP::BOARD_BROADCAST_ADDR);
      stream->write(config_byte);
    }

//---------- Button Illumination Control -----------
    
//-- setLED - set individual LED

    void setLED( byte board_address, byte led_index, lamp_fcn fcn, color_code color )
    {
      if (board_address != 16) // pass thru broadcast address unchanged
      { if (board_address >= board_count) return; // failsafe
        board_address = (board_count-1) - board_address; // tx addresses are reversed
      }
      led_index &= 0b1111;
      stream->write((byte)0xA0 + board_address);
      stream->write((fcn << 4) + led_index);
      stream->write((uint8_t)color);
    }

//-- setLEDrow - set a row of LEDs

    void setLEDrow( byte board_address, byte row_index, lamp_fcn fcn, color_code color )
    {
      row_index &=   0b11;
      if (board_address != 16) // pass thru broadcast address unchanged
      { if (board_address >= board_count) return; // failsafe
        board_address = (board_count-1) - board_address; // tx addresses are reversed
      }
      stream->write((byte)0xA0 + board_address);
      stream->write((fcn << 4) + row_index);
      stream->write((PAT_ROW << 4) + (uint8_t)color);
    }

//-- setLEDcol - set a column of LEDs

    void setLEDcol( byte board_address, byte col_index, lamp_fcn fcn, color_code color )
    {
      col_index &=   0b11;
      if (board_address != 16) // pass thru broadcast address unchanged
      { if (board_address >= board_count) return; // failsafe
        board_address = (board_count-1) - board_address; // tx addresses are reversed
      }
      stream->write((byte)0xA0 + board_address);
      stream->write((fcn << 4) + col_index);
      stream->write((PAT_COL << 4) + (uint8_t)color);
    }

//-- setLEDall - set all LEDs

    void setLEDall( byte board_address, lamp_fcn fcn, color_code color )
    {
      if (board_address != 16) // pass thru broadcast address unchanged
      { if (board_address >= board_count) return; // failsafe
        board_address = (board_count-1) - board_address; // tx addresses are reversed
      }
      stream->write((byte)0xA0 + board_address);
      stream->write((fcn << 4) + 0);
      stream->write((PAT_ALL << 4) + (uint8_t)color);
    }
};

#endif
