//--------------------------------------------
// LoRa Remote Control
// RSP 2021.11.01
// Target: ATmega328PB 3.3V/12M (custom PCB) [board = Pololu A-Star 328PB]
//       + SX1278 radio
//--------------------------------------------

#include <avr/sleep.h>

#include <LoRa.h> // default pins are: cs=10, reset=9, irq=2
#define NETID 0xB4 // ranges from 1-255, default 0x34
#define MYID    0  // set for each remote 0..15
#define RESENDS 5  // nbr of redundant packets to send

#include <FastLED.h>
#define NUM_LEDS 1
CRGB leds[NUM_LEDS];

#define BAT_PIN A0 // 4.2Vmax on 20K:20K voltage divider
#define BTN_PIN 3  // button, netagive logic
#define VOL_PIN A5 // volume knob
#define SEL_PIN A4 // select knob
#define PIX_PIN 5

uint8_t selection;

//--------------------------------------------

boolean checkBatt()
{
  // Aref = 3.3V, 1:2 voltage divider
  //   V = adc*6.6/1023
  // min = 3.5V cutoff => 3.5/6.6*1023 = 542.5 adc
  return (analogRead(BAT_PIN) > 542);
}

void lowBatteryShutdown()
{
Serial.println("SHUTDOWN"); Serial.flush();  
  LoRa.sleep(); // power down radio
  // flash low battery
  for (uint8_t i=0; i < 10; i++)
  { leds[0] = 0x000008; FastLED.show();
    delay(250);
    leds[0] = 0; FastLED.show();
    delay(500);
  }  
  // shut down CPU
  do // forever
  { sleep_enable();
    set_sleep_mode(SLEEP_MODE_PWR_DOWN);
    sleep_cpu();
  } while (1);
  // the neopixel remains powered on but dark, consuming ~1mA
}

//--------------------------------------------
void paintLED( uint8_t color )
{
  switch (color)
  { case 0: leds[0] = 0x000010; break;
    case 1: leds[0] = 0x001000; break;
    case 2: leds[0] = 0x100000; break;
    case 3: leds[0] = 0x101000; break;
    case 4: leds[0] = 0x100010; break;
    case 5: leds[0] = 0x001010; break;
    default:leds[0] = 0x101010; break;
  }
  FastLED.show();    
}

void setup() 
{
  Serial.begin(9600); // diagnostics
  
  pinMode(BTN_PIN, INPUT_PULLUP );

  FastLED.addLeds<WS2811, PIX_PIN, RGB>(leds, NUM_LEDS);
  leds[0] = 0x001000; // green
  FastLED.show();
  
  // all radios on same network (sync word)
  if (!LoRa.begin(433E6)) 
    while (1) { leds[0] = 0x100000; FastLED.show(); delay(250); leds[0] = 0; FastLED.show(); delay(250); } 
  LoRa.setSyncWord(NETID);

  // initial selection
  selection = map( analogRead(SEL_PIN), 0,1023, 0,6 );
  paintLED(selection);
}

//--------------------------------------------

void loop() 
{

//-- monitor battery

#define BATT_MONITOR_RATE 1000UL // test battery once per second
if (1)  
  { static uint32_t batt_tm = millis();
    if (millis() - batt_tm > BATT_MONITOR_RATE)
    { batt_tm = millis();
      if (!checkBatt()) lowBatteryShutdown(); // never returns (requires power cycle)
    }
  }

//-- monitor select knob

  { int16_t pot = analogRead(SEL_PIN);
    uint8_t n = map( pot, 0,1023, 0,6 );
    if (n != selection)
    { const int deadband = 16;
      const int band = 1024/6;
      const int half = 1024/12;
      int16_t cp = selection*band+half;
      if (pot > cp+half+deadband/2 || pot < cp-half-deadband/2)
        paintLED(selection = n);
    }      
  }

//-- monitor button

  if (!digitalRead(BTN_PIN))
  {
    uint8_t v = map( analogRead(VOL_PIN), 0,1023, 0,15);
    uint8_t x = selection + (v << 4);
    LoRa.beginPacket();
      LoRa.write(x);
    LoRa.endPacket();
    for (uint8_t i=200; i; i--) { delay(1); if (!digitalRead(BTN_PIN)) i=200; }
  }
}
