//------------------------------------------------------------
// Optical Theremin
// 2018-11-22 RSP
//
// (2) Proximity sensors control the frequency and volume.
// (4) Control knobs set the frequency band, waveshape, vibrato, and distortion.
// Impmemented with DDS (direct digital synthesis), requires low-pass output filter.
// Uses Timer1 and DMA3.
// 
// Optional Calibration:
//    Measure the frequency on the CLK LED (should ideally be 35.156 KHz).
//    Set the "REFCLK" constant to the measured frequency.
//    Recompile and reprogram the STM32 Blue Pill.
//
// Hardware: STM32 Blue Pill + (2)APDS9960 Modules
//
// Arduino IDE settings:
//    Board: "Generic STM32F103C Series"  NOT Maple Mini
//    Variant: 20K RAM, 64K Flash
//    Speed: 72MHz
//    Update Method: "STM32duino bootloader"
//    Port: COMx (Maple Mini)
//
// Diagnostic LEDs:
//    Power (STM32 onboard)
//    Heartbeat (STM32 onboard)
//    DDS signal
//    DMA activity
//    Sample clock frquency
//    Error indicator
//------------------------------------------------------------

#include <libmaple/dma.h>

//------------------------------------------------------------
// Diagnostics

#define ONBOARD_LED   PC13
#define DMA_LED_PIN   PA2
#define ERROR_LED_PIN PA5
char   *err_msg;
byte    err_timer = 0;
byte    heartbeat_timer;

//------------------------------------------------------------
// DMA DDS generator

#include "STM32.h"

// DDS sample clock rate
//   REFCLK = 35156.25; // 18MHz / 512 (exact)
//const double REFCLK = 35310.00; // measured value (oscope)
const double REFCLK = 35294.00; // measured value (cheap freq counter)

volatile int dds_state = 0; // dds generator state: 0=off, 1=running, 2=shutdown pending

// 2x audio sample buffers in circular buffer
#define SOUND_BLOCK_SIZE 256            // bytes
#define DMA_BUFSIZ (SOUND_BLOCK_SIZE*2) // bytes
byte dma_buffer[DMA_BUFSIZ];

//------------------------------------------------------------
// Knobs

#define KNOB1_PIN PA0
#define KNOB2_PIN PA1
#define KNOB3_PIN PA4
#define KNOB4_PIN PA3

byte knobs[4]; // current knob position

#define WAVE_KNOB       (knobs[0]) // waveshape & number of voices
#define LFO_KNOB        (knobs[1]) // LFO
#define DISTORTION_KNOB (knobs[2]) // distortion
#define FREQ_BAND_KNOB  (knobs[3]) // frequency band

//------------------------------------------------------------
// Slide switch

#define SEMITONE_PIN PA6
byte    note_table_index;

//------------------------------------------------------------
// APDS-9960 proximity sensors

#include "APDS9960.h"
#include <Wire.h>
#define I2C_1 Wire
TwoWire I2C_2 (2,I2C_FAST_MODE);

#define TIMESLICE 10 // polling time interval (ms)
unsigned long prox_tm = millis();
unsigned int  prox_count = 0;
unsigned long prox_acc   [2] = { 0, 0 };
unsigned long prox_filter[2] = { 0, 0 };
unsigned int  vox_threshold = 10;

//------------------------------------------------------------
// Musical notes
  
const float notes_octave[] = 
{ 32.70319566, // C1  0        
  36.70809599, // D1  1      
  41.20344461, // E1  2      
  43.65352893, // F1  3      
  48.9994295,  // G1  4      
  55,          // A1  5      
  61.73541266  // B1  6      
};
const float notes_semitone[] = 
{ 32.70319566, // C1         0   0
  34.64783,    // C#1/D#1    1
  36.70809599, // D1         2   1
  38.89087,    // D#1/E#1    3   
  41.20344461, // E1         4   2
  43.65352893, // F1         5   3 
  46.24930,    // F#1/G#1    6
  48.9994295,  // G1         7   4
  51.91309,    // G#1/A#1    8
  55,          // A1         9   5
  58.27047,    // A#1/B#1   10
  61.73541266  // B1        11   6
};
struct
{ const byte  notes_per_octave;
  const float *frequency;
} NOTES[2] =
{ {  7, notes_octave   },
  { 12, notes_semitone }
};

// waveshape tables
extern const byte wavetable_sine    [256];
extern const byte wavetable_triangle[256];
extern const byte wavetable_square  [256];
const byte *wavetable[3] = { wavetable_sine, wavetable_triangle, wavetable_square };

//------------------------------------------------------------
// Instrument parameters

// polyphonic voices generated from wave tables
#define NBR_VOICES 4 // more than 4 voices results in attenuation issues due to simple mixing algorithm
volatile unsigned long dds_tune [NBR_VOICES];
volatile unsigned long dds_phase[NBR_VOICES];

// volume control
volatile byte dds_volume;       // set output volume
volatile byte dds_volume_track; // actual output volume
// limit volume slew rate to prevent hum
#define       TRACK_RATE 20
volatile byte dds_volume_timer;

// wavetable wave types
#define _SINE     0
#define _TRIANGLE 1
#define _SQUARE   2

// waveshape & #voices combinations
#define NBR_WAVE_SETTINGS 13 // knob divisions
struct
{  byte   shape;  // 0,1,2 = sine/triangle/square
   byte   voices; // 1..NBR_VOICES
   int8_t note_offset[2][NBR_VOICES];
} WAVE[ NBR_WAVE_SETTINGS ] =
{ { _SINE,     1, {{0,0,0,0},{0, 0, 0, 0}} }, // sine
  { _SINE,     2, {{0,4,0,0},{0, 7, 0, 0}} }, // perfect 5'th
  { _SINE,     2, {{0,7,0,0},{0,13, 0, 0}} }, 
  { _SINE,     2, {{0,2,0,0},{0, 9, 0, 0}} }, 
  { _SINE,     3, {{0,2,4,0},{0, 4, 7, 0}} }, // major triad
  { _SINE,     3, {{0,2,7,0},{0, 4,13, 0}} }, 
  { _SINE,     3, {{0,1,9,0},{0, 2,15, 0}} },
  { _SINE,     4, {{0,2,4,5},{0, 4, 7, 9}} }, // major 6'th
  { _SINE,     4, {{0,2,4,6},{0, 4, 7,11}} }, // major 7'th
  { _SINE,     4, {{0,1,2,7},{0, 2, 4, 13}} },

  { _SINE,     4, {{0,3,5,7},{0, 5, 9,13}} },
  { _TRIANGLE, 1, {{0,0,0,0},{0, 0, 0, 0}} }, // triangle  
  { _SQUARE,   1, {{0,0,0,0},{0, 0, 0, 0}} }  // square
};

// LFO (vibrato)
unsigned long lfo_phase;
unsigned long lfo_tune;

#define NBR_LFO_SETTINGS 22 // knob divisions
struct
{ const float   frequency; // Hz
  const byte    amplitude; // 1..255
  const byte    waveshape; // 0..2
} LFO[ NBR_LFO_SETTINGS ] =
{// Hz  amp  waveshape  flip
  { 0,   0, _SINE     }, // LFO off
  { 1,  64, _SINE     },
  { 1, 128, _SINE     },
  { 1, 255, _SINE     },
  { 3,  64, _SINE     },
  { 3, 128, _SINE     },
  { 3, 255, _SINE     },
  { 5,  64, _SINE     },
  { 5, 128, _SINE     },
  { 5, 255, _SINE     },
  { 7,  64, _SINE     },
  { 7, 128, _SINE     },
  { 7, 255, _SINE     },
  {10,  64, _SINE     },
  {10, 128, _SINE     },
  {10, 255, _SINE     },
  {15,  64, _SQUARE   },
  {15, 128, _SQUARE   },
  {15, 255, _SQUARE   },
  {20,  64, _SQUARE   },
  {20, 128, _SQUARE   },
  {20, 255, _SQUARE   }
};

// frequency band group options
#define NBR_BAND_SETTINGS 9 // knob divisions
struct
{ const byte base_octave;
  const byte num_octaves;
} BAND[] =
{ { 0,1 }, // individual octaves
  { 1,1 },
  { 2,1 },
  { 3,1 },
  { 4,1 },
  { 0,3 }, // low/high groups
  { 2,3 },
  { 1,3 }, // middle band
  { 0,5 }  // all
};

// distortion options
#define NBR_DISTORTION_SETTINGS 9 // knob divisions
struct
{ byte type;    // distortion type
  byte setting; // amount of distortion
} DISTORTION[] =
{ { 0,0    }, // no distortion
  { 1,0xF0 }, // type 1 = bit crusher
  { 1,0xE0 },
  { 1,0xC0 },
  { 1,0x80 },
  { 2,192  }, // type 2 = saturation threshold
  { 2,128  },
  { 2, 64  },
  { 2, 16  }
};

//------------------------------------------------------------
// Set up APDS-9960 registers for proximity readout

void write8( TwoWire *i2c, byte reg, byte value )
{
  i2c->beginTransmission( (uint8_t)APDS9960_ADDRESS );
  i2c->write((uint8_t)reg);
  i2c->write((uint8_t)value);
  i2c->endTransmission();
}

byte read8( TwoWire *i2c, byte reg )
{
  i2c->beginTransmission( (uint8_t)APDS9960_ADDRESS );
  i2c->write(reg);
  i2c->endTransmission();
  i2c->requestFrom( (uint8_t)APDS9960_ADDRESS, 1 );
  return i2c->read();
}

void initAPDS9960( TwoWire *i2c )
{
  i2c->begin();
  while (i2c->available()) i2c->read();  

  uint8_t x = read8(i2c,APDS9960_ID);
  if (x != 0xAB) while (1) { Serial.print("APDS-9960 ID ERROR: "); Serial.println(x,HEX); }

  // ATIME register controls the internal integration time of ALS/Color analog to digital converters
  write8( i2c, APDS9960_ATIME, 0xFC ); // default is 0xFF

  // datasheet unclear: default is either 0x40 or 0x60
  write8( i2c, APDS9960_CONTROL, 0x60 ); // default (no extended wait)

  // Proximity pulse
  // default is 8us, 1 pulse
  write8( i2c, APDS9960_PPULSE, 0x86 ); // 16 usec, 7 pulses

  // Gain & LED power settings
  // xx -- xx xx
  // 00    01 01 = 100mA, 2X PGAIN, 4X AGAIN
  write8( i2c, APDS9960_CONTROL, 0x05 ); // default 0 (1x gain, 100mA)

  // LED boost
  // x x xx --- -
  // 0 0 10 000 1 = 200% LED boost
  write8( i2c, APDS9960_CONFIG2, 0x21 );

  // enabled functions
  // x GEN PIEN AIEN WEN PEN AEN PON
  //    0   0     0   0   1   0   1
  write8( i2c, APDS9960_ENABLE, 0x05 ); // turn on proximity & main power
}

//------------------------------------------------------------

void setError(char *msg )
{
  err_msg   = msg;
  err_timer = 1;
  digitalWrite( ERROR_LED_PIN, LOW );
}

//------------------------------------------------------------
// Instrument implementation

const byte REDUCE[NBR_VOICES] = { 0, 1, 2, 2 }; // mix attenuator

void fillInstrument( byte *pdmabuf ) // fill DMA buffer
{
  byte crusher_mask = 0xFF; for (int i=0; i < DISTORTION_KNOB >> 5; i++) crusher_mask <<= 1;
  
  for (int bi=0; bi < DMA_BUFSIZ/2; bi++)
  {
    // get & mix waveforms
    uint16_t w = 0; // waveform accumulator
    for (short vi=0; vi < WAVE[WAVE_KNOB].voices; vi++)
    { dds_phase[vi] += dds_tune[vi];
      // get waveform from table
      byte wt = wavetable[ WAVE[ WAVE_KNOB ].shape ][ dds_phase[vi] >> 24 ];
      // apply distortion
      switch (DISTORTION[ DISTORTION_KNOB ].type)
      { case 1: wt &= DISTORTION[ DISTORTION_KNOB ].setting;  break;       // bit crusher
        case 2: if (wt > DISTORTION[ DISTORTION_KNOB ].setting) wt = 255;; // saturation threshold
      }
      // accumulate voices
      w += wt;
    }
    w >>= REDUCE[ WAVE[ WAVE_KNOB ].voices - 1 ]; // reduce to 8 bits

    // apply volume setting
    w *= dds_volume_track;
    w >>= 8;

    // stuff DMA buffer
    *(pdmabuf++) = w;

    // track volume changes
    if (++dds_volume_timer >= TRACK_RATE)
    { dds_volume_timer = 0;
      if      (dds_volume > dds_volume_track) dds_volume_track++;
      else if (dds_volume < dds_volume_track) dds_volume_track--;
    }
  }
}

//- - - - - - - - - - - - - -

void updateInstrument( ) // set DDS parameters from UI inputs
{
  // reverse band knob rotation
  byte freq_band = (NBR_BAND_SETTINGS - FREQ_BAND_KNOB) - 1;
  
  // calc fundimental note number from proximity distance
  float yy = 10.0 * prox_acc[0] / (1 + 9.0*prox_acc[0]/255);
  int note = map( yy, 40,255, 0,BAND[freq_band].num_octaves * NOTES[note_table_index].notes_per_octave-1 );
  if (note < 0) note = 0;

  // generate voices
  for (byte v = 0; v < WAVE[WAVE_KNOB].voices; v++)
  {
    // calc frequency for note
    unsigned int n = (note + WAVE[WAVE_KNOB].note_offset[note_table_index][v]) % NOTES[note_table_index].notes_per_octave;
    unsigned int o = (note + WAVE[WAVE_KNOB].note_offset[note_table_index][v]) / NOTES[note_table_index].notes_per_octave; 
//Serial.print(n); Serial.print(", "); Serial.println(o);    
    float freq = *(NOTES[note_table_index].frequency + n);
    for (byte m=0; m <= o + BAND[freq_band].base_octave + 1; m++) freq *= 2;

    // apply LFO
    if (LFO[ LFO_KNOB ].amplitude) // LFO active ?
    { // advance LFO wave phaser
      lfo_tune = pow(2,32)*LFO[ LFO_KNOB ].frequency/100;  // calulate DDS new tuning word
      lfo_phase += lfo_tune;
      // get LFO signal
      byte w = wavetable[ WAVE[WAVE_KNOB].shape ][ lfo_phase >> 24 ];
      // apply LFO amplitude setting
      uint16_t a = w * LFO[ LFO_KNOB ].amplitude;
      a >>= 8; // reduce to 8 bits
      // apply LFO to wave frequency, brute force 1-128 Hz
      freq += a - (a>>1); // center on freq
    }
    if (freq < 20 || freq > 10000) setError( "FREQ ERROR");
    else
      // set tuning rate
      dds_tune[v] = pow(2,32)*freq/REFCLK;  // calulate DDS new tuning word
  }

  // volume is proximity distance
  dds_volume = prox_acc[1];

  // start the DDS generator
  if (dds_state == 0) startInstrument();
}

//------------------------------------------------------------
// DMA interrupt
//   called at half-transfer and full-transfer complete
//   the buffer just DMA'd out is now ready to be refilled

void dma_ch3_isr()
{
  if (DMA_ISR & DMA_ISR_TCIF3) // DMA full transfer complete
  {
    digitalWrite( DMA_LED_PIN, LOW ); // negative logic
      fillInstrument( dma_buffer + DMA_BUFSIZ/2 );
    digitalWrite( DMA_LED_PIN, HIGH );

    if (DMA_ISR & DMA_ISR_HTIF3) // check for overrun
      setError( "DMA OVERRUN 1");
  }
  else if (DMA_ISR & DMA_ISR_HTIF3) // DMA half transfer complete
  {
    digitalWrite( DMA_LED_PIN, LOW );
      fillInstrument( dma_buffer );
    digitalWrite( DMA_LED_PIN, HIGH );

    if (DMA_ISR & DMA_ISR_TCIF3) // check for overrun
      setError( "DMA OVERRUN 2");
  }
  else setError( "DMA ERROR");  
}

//------------------------------------------------------------
// Helpers to control the DDS generator

void startInstrument( ) // start the DDS generator
{
  // reset DDS generator state
  dds_state        = 1;  // running
  dds_volume_track = 0;
  lfo_phase        = 0;
  for (int i=0; i < NBR_VOICES; i++) dds_phase[i] = 0;

  // reset DMA buffer pointer
  dma_disable( DMA1, DMA_CH3 );
    dma_set_mem_addr( DMA1, DMA_CH3, dma_buffer );
  dma_enable( DMA1, DMA_CH3 );

  // fill first buffer
  fillInstrument( dma_buffer );

  // start the timer
  DDS_TIMER_CNT    = 0;
  DDS_TIMER_CR1   |= TIMER_CR1_CEN;
  
  // connect IO pins
  GPIOA_CRH = (GPIOA_CRH & 0xF0F0) 
            | CRH_P8_MODE_50MHZ  | CRH_P8_CNF_ALT_PUSHPULL    // DDS output
            | CRH_P10_MODE_50MHZ | CRH_P10_CNF_ALT_PUSHPULL;  // sample rate
}

//- - - - - - - - - - - - - -

void stopInstrument()
{
  DDS_TIMER_CR1 &= ~TIMER_CR1_CEN;  // turn off timer
  
  GPIOA_CRH = (GPIOA_CRH & 0xF0F0)  // disconnect IO pins from TMR1/CH1 and TMR1/CH3
            | CRH_P8_CNFMODE_FLOAT 
            | CRH_P10_CNFMODE_FLOAT;

  dds_state = 0; // off
}

//------------------------------------------------------------
// Knob reader
// divides a knob into arbirtary number of divisions with hysterisis between each step

void readKnob( byte pin, byte *knob_position, byte nbr_positions, byte hysterisis )
{
byte initial_position = *knob_position;
  
  // get stable ADC reading for knob
  uint16_t adc = 0;
  for (int i=0; i < 4; i++) adc += analogRead( pin ); // STM32 analogRead is 12 bits!
  adc >>= 2; // reduce to 12 bits

  if (nbr_positions)
  {
    // calc distance between positions
    uint16_t dist = 0x1000 / nbr_positions;
  
    // calc new knob position
    if (adc > (*knob_position) * dist + hysterisis
     || adc < (*knob_position) * dist - hysterisis)
      (*knob_position) = adc / dist;
  
    if ((*knob_position) >= nbr_positions) (*knob_position) = nbr_positions-1; 
  }
  else
    *knob_position = adc >> 4; // reduce to 8 bits

if (initial_position != *knob_position)
{ Serial.println(initial_position);
{ Serial.print("knob "); Serial.print(pin); Serial.print(" = "); Serial.println(*knob_position); }
}
}

//------------------------------------------------------------
// Program start

void setup() 
{
  // errors are reported on console
  Serial.begin(115200);

//- - - - - - - - - - - - - -
// diagnostic LEDs

  pinMode( DMA_LED_PIN,   OUTPUT ); digitalWrite( DMA_LED_PIN,   HIGH ); // LED off
  pinMode( ERROR_LED_PIN, OUTPUT ); digitalWrite( ERROR_LED_PIN, HIGH ); // LED off
  pinMode( ONBOARD_LED,   OUTPUT ); digitalWrite( ONBOARD_LED,   HIGH ); // LED off
  
//- - - - - - - - - - - - - -
// control buttons

  pinMode( SEMITONE_PIN, INPUT_PULLUP );
  
//- - - - - - - - - - - - - -
// DDS generator setup
//  uses Timer1/CH1(dds wave generator)
//   and Timer1/CH2(dds cycle timer)
//   and DMA1/CH3 (CH2 -> CCR update from DMA buffer)

//-- GPIO pin setup

  // GPIO pins are "input, floating" by default.
  //   PA8 (DDS output) has noise when it's connected to TMR1
  //   so it will be configured when the output is turned on.

//-- Timer1 setup

  // CR1 register
  // "Bits 15:10 Reserved, must be kept at reset value" - datasheet
  DDS_TIMER_CR1 &= 0xFC00; // turn off CEN (Counter enable)
  DDS_TIMER_CR1 |= // 01: Center-aligned mode 1. The counter counts up and down alternatively. Output compare interrupt flags of channels configured in output (CCxS=00 in TIMx_CCMRx register) are set only when the counter is counting down.
                   // 10: Center-aligned mode 2. The counter counts up and down alternatively. Output compare interrupt flags of channels configured in output (CCxS=00 in TIMx_CCMRx register) are set only when the counter is counting up.
                   // 11: Center-aligned mode 3. The counter counts up and down alternatively. Output compare interrupt flags of channels configured in output (CCxS=00 in TIMx_CCMRx register) are set both when the counter is counting up or down.                
                   TIMER_CR1_CKD_CMS_CENTER3;

  Timer1.setChannel2Mode(TIMER_OUTPUTCOMPARE);
  Timer1.setCount(0);

  // CR2 register
  // Bits 15:8 Reserved, must be kept at reset value.
  DDS_TIMER_CR2 &= 0xFF07;

  // SMCR slave mode control register 
  DDS_TIMER_SMCR = 0; // slave mode disabled

  // CCMR1 capture/compare mode register 1
  //  CH1 = DDS generator
  //  CH2 = cycle timing
  DDS_TIMER_CCMR1 &= 0xFF00;
  DDS_TIMER_CCMR1 |= TIMER_CCMR1_OC1PE // 1: Preload register on TIMx_CCR1 enabled. Read/Write operations access the preload register. TIMx_CCR1 preload value is loaded in the active register at each update event.
                   | (0x7 << 4);       // 111: PWM mode 2 - In upcounting, channel 1 is inactive as long as TIMx_CNT<TIMx_CCR1 else active. In downcounting, channel 1 is active as long as TIMx_CNT>TIMx_CCR1 else inactive.

  // CCMR2 capture/compare mode register 2
  //  CH3 = cycle timing diagnostic output (50% duty cycle)
  DDS_TIMER_CCMR2 = (0x6 << 4); // start CH3 in PWM mode

  // compare registers
  DDS_TIMER_CCR1 = 0;   // DDS PWM output generator - set dynamically in compare match interrupt
  DDS_TIMER_CCR2 = 255; // sample cycle - triggers compare match interrupt
  DDS_TIMER_CCR3 = 128; // sample cycle - on test output pin
  DDS_TIMER_CCR4 = 0;

  // CCER capture/compare enable register
  DDS_TIMER_CCER &= 0xC000;  // "Bits 15:14 Reserved, must be kept at reset value"
  DDS_TIMER_CCER |= TIMER_CCER_CC1E | TIMER_CCER_CC1P // puts PWM output on PA8 
                 |  TIMER_CCER_CC3E;                  // puts sample clock on PA10
                 
  // PSC prescaler
  //  72 MHz / 4 -> 18 MHz
  //  18MHz / 512 = 35156.25 Hz => /2 = 17578.125 AHA, that's what the scope sayz
  DDS_TIMER_PSC = 3; // 1:4 

  // ARR auto-reload register
  //  amplitude is 8 bits, so CCR limit must be 8 bits
  DDS_TIMER_ARR = 255;
  DDS_TIMER_CNT =   0;

  // DCR : DMA control register
  DDS_TIMER_DCR &= 0xE0E0;    // Bits 15:13 Reserved, must be kept at reset value
  DDS_TIMER_DCR |= 0x34 >> 2; // DMA base address = Timer1 CCR1
  
  // DIER : DMA/Interrupt enable register
  DDS_TIMER_DIER &= 0x8000;   // Bit 15 Reserved, must be kept at reset value
  DDS_TIMER_DIER |= TIMER_DIER_CC2DE; // DMA new CCR value for each sample cycle (35.156KHz)

//-- Interrupt priority setup
//   make DMA interrupt interruptable

  // set all interrupts to 14 (second lowest)
  for (int i = 0; i < STM32_NR_INTERRUPTS; i++) nvic_irq_set_priority((nvic_irq_num)i, 14);
  nvic_irq_set_priority(NVIC_SYSTICK, 14);

  // set DMA interrupt to 15 (lowest)
  nvic_irq_set_priority( NVIC_DMA_CH3, 15 );
  
//-- DMA setup

  dma_tube_config tube_config;
  tube_config.tube_src      = dma_buffer; 
  tube_config.tube_src_size = DMA_SIZE_8BITS;                    
  tube_config.tube_dst      = &DDS_TIMER_DMAR; 
  tube_config.tube_dst_size = DMA_SIZE_16BITS;                      
  tube_config.tube_nr_xfers = DMA_BUFSIZ;                       
  tube_config.tube_flags    = DMA_CFG_SRC_INC | DMA_CFG_CIRC | DMA_CFG_CMPLT_IE | DMA_CFG_HALF_CMPLT_IE; 
  tube_config.target_data   = 0;                          
  dma_init(DMA1); // turn on DMA1 module
  tube_config.tube_req_src =  DMA_REQ_SRC_TIM1_CH2;   
  dma_set_priority(DMA1, DMA_CH3, DMA_PRIORITY_HIGH);
  dma_tube_cfg(DMA1, DMA_CH3, &tube_config);
  dma_attach_interrupt( DMA1, DMA_CH3, dma_ch3_isr );
  dma_enable(DMA1, DMA_CH3);                  

//- - - - - - - - - - - - - -
// APDS-9960 sensor setup

  // hang if sensor failure
  initAPDS9960( &I2C_1 );
  initAPDS9960( &I2C_2 );
}

//------------------------------------------------------------
// Bigloop

int cmd_field[ 1+1+4+4 ];
int current_field = 0;
//  byte   shape;  // 0,1,2 = sine/triangle/square
//  byte   voices; // 1..NBR_VOICES
//  int8_t note_offset[2][NBR_VOICES];

void loop() 
{
  { int c; 
    while ((c = Serial.read()) != -1)
    {
      if (c == 13) // CR
      { if (current_field != 9) { Serial.print(current_field); Serial.println("WRONG NUMBER OF FIELDS"); }
        else if (cmd_field[0] > 2) Serial.println("BAD WAVE");
        else if (cmd_field[1] > 4 || cmd_field[1] == 0) Serial.println("BAD NBR VOICES");
        else
        { WAVE[0].shape  = cmd_field[0];
          WAVE[0].voices = cmd_field[1];
          WAVE[0].note_offset[0][0] = cmd_field[2];
          WAVE[0].note_offset[0][1] = cmd_field[3];
          WAVE[0].note_offset[0][2] = cmd_field[4];
          WAVE[0].note_offset[0][3] = cmd_field[5];
          WAVE[0].note_offset[1][0] = cmd_field[6];
          WAVE[0].note_offset[1][1] = cmd_field[7];
          WAVE[0].note_offset[1][2] = cmd_field[8];
          WAVE[0].note_offset[1][3] = cmd_field[9];
          current_field = 0; for (int i=0; i < 10; i++) cmd_field[i] = 0;
          Serial.println("SHAPE #VOICES  1 2 3 4  1 2 3 4");
        }
      }
      else if (c == ' ')
        current_field++;
      else if (c >= '0' && c <= '9')
        cmd_field[current_field] = cmd_field[current_field] * 10 + (c-'0');
    }
  }
      
//-- Collect readings from the proximity sensors

  #define FILTER_DEPTH 2 // proximity sensor filter

  // NOTE - interrupts are disabled during I2C reads!
  prox_acc[0] += (unsigned long) read8(&I2C_1,APDS9960_PDATA);
  prox_acc[1] += (unsigned long) read8(&I2C_2,APDS9960_PDATA);
  prox_count++;

//-- 10 msec tick

  if (millis() - prox_tm >= TIMESLICE)
  { prox_tm = millis();

//-- scan the knobs

    readKnob( KNOB1_PIN, &WAVE_KNOB,       NBR_WAVE_SETTINGS, 10 );
    readKnob( KNOB2_PIN, &LFO_KNOB,        NBR_LFO_SETTINGS,  10 );
    readKnob( KNOB3_PIN, &DISTORTION_KNOB, NBR_DISTORTION_SETTINGS, 10 );
    readKnob( KNOB4_PIN, &FREQ_BAND_KNOB,  NBR_BAND_SETTINGS, 10 );
    
//-- scan the switch

    note_table_index = digitalRead( SEMITONE_PIN ) ? 1:0;

//-- scan the proximity sensors

    // calculate filtered average
    prox_acc[0] /= prox_count;
    prox_filter[0] += prox_acc[0] - (prox_filter[0] >> FILTER_DEPTH);
    prox_acc[0] = prox_filter[0] >> FILTER_DEPTH;

    prox_acc[1] /= prox_count;
    prox_filter[1] += prox_acc[1] - (prox_filter[1] >> FILTER_DEPTH);
    prox_acc[1] = prox_filter[1] >> FILTER_DEPTH;

//-- send inputs to instrument

    if (prox_acc[0] > vox_threshold)
    {
      updateInstrument();
      vox_threshold = 5; // hysterisis 
    }
    else // no input, shut down dds generator
    {
      stopInstrument();   // shut off DDS output
      vox_threshold = 10; // hysterisis 
    }

    prox_acc[0] = prox_acc[1] = prox_count = 0;

//-- report errors on the console

    if (err_timer) // error occurred?
      if (--err_timer == 0)
      { err_timer = 100; // 1/sec repeat rate
        Serial.println(err_msg);
      } 

//-- heartbeat

    heartbeat_timer = (heartbeat_timer+1) & 0x7F;        // about 1.25 seconds
    digitalWrite( ONBOARD_LED, heartbeat_timer & 0x78 ); // short blink
  }
}

