//------------------------------------------------------
// Teensy Drone UI
// 2023.05.20 RSP
// Target: RP2040
//         MIDI IN:  control input to RP2040
//         MIDI OUT: headless Teensy4 synth
//         ILI9341 + XPT2046 touch (touch not used)
//         (7) VelociRotor knob chain
//         (1) status LED (onboard)
//
// Description
//   Teensy synth user interface
//   MIDI controllable
//
// Navigation
//   Splash Page
//   Mode pages = synth controls
//   About Page = info & global controls
//   Navigation dial
//     spin to select page
//     press for About
//     hold to toggle MIDI value display
//------------------------------------------------------

//#define DEBUG_CONSOLE

#include "common.h" // things in common with Teensy codebase

//------------------------------------------------------
// database

#include <vector>        // dynamic array
#include <unordered_set> // dynamic binary tree

#include "app.h"         // constants
EEdatabase DB;

//------------------------------------------------------
// knobs

#include "FEARLESS_VelociBus.h"
VelociBus vbus;   // create a VelociBus interface

//------------------------------------------------------
// 240x320 Display (ILI9341+XPT2046)
/* 
  Library customizations...
    User_Setup_Select.h
      select User_Setups/Setup60_RP2040_ILI9341.h
      comment out //#include <User_Setup.h>           // Default setup is root library folder
    Setup60_RP2040_ILI9341.h
      select ILI9341_DRIVER
      Set SPI_FREQUENCY 10-32MHz seems to work on breadboard
      #define TFT_BACKLIGHT_ON HIGH  // Level to turn ON back-light (HIGH or LOW)
      // hardware SPI pins
      #define TFT_MISO 16  // SPI0
      #define TFT_MOSI 19
      #define TFT_SCLK 18
      // GPIO pins
      #define TFT_CS   17  // Chip select control pin
      #define TFT_DC    3  // Data Command control pin
      #define TFT_RST  20  // Reset pin (could connect to Arduino RESET pin)
      #define TFT_BL    2  // LED back-light
      #define TOUCH_CS 21  // Chip select pin (T_CS) of touch screen
*/
#include "Free_Fonts.h" // Include the header file attached to this sketch
#include "SPI.h"
#include "TFT_eSPI.h"
TFT_eSPI tft = TFT_eSPI(); // Using hardware SPI

#include "pages.h"
PageController page_controller;

//------------------------------------------------------
// MIDI interface

#include <MIDI.h>
#define MIDI_SERIAL Serial1
// MIDI DIN
//   input is from MIDI program (optional)
//   output is to T4 synth
MIDI_CREATE_INSTANCE(HardwareSerial, MIDI_SERIAL, MIDI); // RP2040 Rx=GP1, Tx=GP0

//------------------------------------------------------
// Processor Core 0 = User Interface
//------------------------------------------------------

// run single core for initialization
volatile boolean hold_core_1 = true;

void setup() 
{
#ifdef DEBUG_CONSOLE  
  // diagnostics
  Serial.begin(115200);
  while (!Serial && millis() < 10000UL);
  Serial.println("core 0 start"); 
#endif

  // status LED
  pinMode( STATUS_LED_PIN, OUTPUT );  digitalWrite( STATUS_LED_PIN, LOW );

  // LCD display
  tft.begin();
  tft.setRotation(2); // portrait
  tft.fillScreen(TFT_BLACK);
  tft.setTextColor(TFT_WHITE, TFT_BLACK);
  analogWrite( BL_PIN, 255); // full brightness to start up

  // knobs
  Serial2.begin(38400); // VelociBus
  vbus.begin( &Serial2 );
  { int count = vbus.commandChainLength(); //chainLength();
DEBUG_VALUE("CHAIN LENGTH = ",count);
    if (count != 7) page_controller.fatalError(BLINK_ERR_VELOCIBUS, "VelociBus "+String(count));
  }
  vbus.rotor9setBrightness( KNOB_TX(KNOB_NAV), 1 ); // min brightness

  // load database from external EEPROM
  DB.loadDatabase(); // to eeprom_data
  // synchronize data
  memcpy( &DB.ui_data, &DB.eeprom_data, sizeof(DB.ui_data) );
  for (uint i=0; i < PATCH_SIZE; i++)
    DB.ui_disp.patch[i] = DB.live_data.patch[i] = 0x80;
 
  // show first page
  page_controller.setBrightness( DB.ui_data.brightness );
  page_controller.start(PG_SPLASH);

  // now safe to start 2'nd core
  hold_core_1 = false; // let setup1 run
}

// - - - - - - - - - - - - -
// core 0 loop

void loop(void) // UI, allowed to block as needed
{

//-- drive the user interface

  // update display from ui, update ui from knobs
  page_controller.run();

//-- automatically save changes to EEPROM after a few seconds

  DB.autoSave();

//-- diagnostic console on serial monitor

#ifdef DEBUG_CONSOLE
  { int c = Serial.read();
    switch (c)
    { case -1 : break;
                    
      case 'S': case 's': // activate screen saver
        page_controller.start( PG_SAVER );
        break;

      case 'E': case 'e': // dump EEPROM
        for (uint i=0; i < 8192*2; i+=32)
        { byte buf[32];
          DB.read( i, buf, 32 );
          Serial.print(i,HEX); Serial.print(":");
          for (uint j=0; j < 32; j++) { Serial.print(" "); if (buf[j] < 16) Serial.print("0"); Serial.print(buf[j],HEX); }
          Serial.println();
        }
    }
  }
#endif  
}

//------------------------------------------------------
// Processor Core 1 = MIDI (fast loop)
//------------------------------------------------------

// MIDI debuggers
void mySendControlChange(byte cc, byte v, byte ch) 
{
#ifdef DEBUG_CONSOLE
Serial.print("ControlChange "); Serial.print(ch); Serial.print(", "); Serial.print(cc); Serial.print(", "); Serial.println(v);
#endif
  MIDI.sendControlChange(cc,v,ch); 
}
void mySendProgramChange(byte v, byte ch) 
{
#ifdef DEBUG_CONSOLE
Serial.print("ProgramChange "); Serial.print(ch); Serial.print(", "); Serial.println(v);
#endif
  MIDI.sendProgramChange(v,ch); 
}


void setup1()
{
  while (hold_core_1) delay(1);
DEBUG_MESSAGE("core 1 start");

  // serial MIDI
  //   IN is from keyboard
  //   OUT is to TSynth

  MIDI_SERIAL.setFIFOSize(1024);
  MIDI.begin(MIDI_CHANNEL_OMNI);
  MIDI.turnThruOff();

  // start midi handlers
  MIDI.setHandleControlChange(myControlChange);
  MIDI.setHandleProgramChange(myProgramChange);
  MIDI.setHandleNoteOff      (myNoteOff);
  MIDI.setHandleNoteOn       (myNoteOn);

  // all systems go
  digitalWrite( STATUS_LED_PIN, HIGH ); // LED on

  // give the Teensy some time to boot
  delay(250);
}

// - - - - - - - - - - - - -
// no blocking loop

void loop1(void)
{
  // handle MIDI input
  if (MIDI.read()) // triggers midi callbacks
  { // diagnostic
    Serial.print("MIDI: ");
    Serial.print(MIDI.getType(),HEX); Serial.print(" ");
    Serial.print(MIDI.getChannel()); Serial.print(" ");
    Serial.print(MIDI.getData1()); Serial.print(" ");
    Serial.println(MIDI.getData2()); Serial.print(" ");
  }
       
  // pace output max 250 messages/sec
  { static uint32_t tm = millis();
    if (millis() - tm < 4) return;
    tm = millis();

    // watch for mode change
    if (DB.live_data.patch[ppMODE] != DB.ui_data.patch[ppMODE])
      { // copy to live and send to synth
        DB.live_data.patch[ppMODE] = DB.ui_data.patch[ppMODE];
        mySendProgramChange(DB.live_data.patch[ppMODE], 1); // send to channel 1
      }

    // watch for CC changes from the UI
    for (uint i=1; i < CC_SIZE; i++)
      if (DB.live_data.patch[i] != DB.ui_data.patch[i])
      { // copy to live and send to synth
        DB.live_data.patch[i] = DB.ui_data.patch[i];
        mySendControlChange(cc_map[i], DB.live_data.patch[i], 1); // send to channel 1
      }
  }
}

// - - - - - - - - - - - - -
// channel filter helper
boolean isMyChannel(byte channel) // returns true if message is for this device
{
  switch (DB.ui_data.channel)
  {
    case  0: return true;  // all
    case 17: return false; // none
    default: // channel 1..16
      return (DB.ui_data.channel == (channel+1));
  }
}

// - - - - - - - - - - - - -
// MIDI CC -> synth & display
void myControlChange(byte channel, byte cc, byte value) // channel 1..16
{
  // qualify channel
  if (!isMyChannel(channel)) return;
  // look up CC
  for (uint i=1; i < CC_SIZE; i++)
    if (cc_map[i] == cc)
    { // update database
      DB.live_data.patch[i] = DB.ui_data.patch[i] = value;
      // send to synth
      mySendControlChange(cc, value, 1);  // forward CC value to synth, always channel 1
      // wake up display
      page_controller.saver_keepalive = true;
    }
}

// - - - - - - - - - - - - -
// MIDI program change -> synth & display
void myProgramChange(byte channel, byte program) // channel 1..16, program 0..127 (constrained to num modes)
{
  if (program >= NUM_MODES) return;
  // qualify channel
  if (!isMyChannel(channel)) return;
  // update database
  DB.live_data.patch[ppMODE] = DB.ui_data.patch[ppMODE] = program;
  // send to synth
  mySendProgramChange(DB.live_data.patch[ppMODE], 1); // send to channel 1
  // wake up display
  page_controller.saver_keepalive = true;
}

// - - - - - - - - - - - - -
// MIDI note on/off -> synth
void myNoteOn([[maybe_unused]]byte inChannel, byte inNoteNumber, byte inVelocity) 
{
  MIDI.sendNoteOn(inNoteNumber,inVelocity,1); // forward to synth 
}
void myNoteOff([[maybe_unused]]byte inChannel, byte inNoteNumber, byte inVelocity)
{ 
  MIDI.sendNoteOff(inNoteNumber,inVelocity,1); // forward to synth
}
