//------------------------------------------------------
// database stored in external EEPROM

#pragma once
#ifndef Lib_DB
#define Lib_DB

#include <Wire.h>
#include "SparkFun_External_EEPROM.h"

// for error reporting
#include <Arduino.h>
extern void fatalErrorHandler( uint blinks, String msg );

#include "common.h"

//------------------------------------------------------
// external EEPROM
//   24C128 = 16K EEPROM will work
//   24C256 = 32K EEPROM is more common, so it's used
//   standard grade EEPROM is ~100K writes PER 64 BYTE PAGE
//     and any write to a page updates the whole page
//
// EEPROM organization, all blocks are 1 page (64 bytes) or less:
//   block contains global variables + CC settings + polarity byte
//   last block is signature

#define AUTOSAVE_MSEC 5000

#define EE_DATABLOCKS 127  // fits in 8K EEPROM
#define EE_SIGNATURE_ADDR (EE_DATABLOCKS*64)
#define EE_SIGNATURE 0x7912

// patch elements
enum patchProperty
{ ppMODE              = 0,
  ppVOLUME            = 1,
  ppOSC1_FREQ         = 2,
  ppOSC1_SH_MOD       = 3,
  ppOSC1_RUNGLER_MOD  = 4,
  ppOSC1_CROSS_MOD    = 5,
  ppOSC2_FREQ         = 6,
  ppOSC2_RUNGLER1_MOD = 7,
  ppOSC2_RUNGLER2_MOD = 8,
  ppOSC2_CROSS_MOD    = 9,
  ppCOMPARE_MIX       = 10,
  ppOSC1_MIX          = 11,
  ppOSC2_MIX          = 12,
  ppFILTER_FREQ       = 13,
  ppFILTER_Q          = 14,
  ppFMOD_RUNGLER1     = 15,
  ppFMOD_OSC2         = 16,
  ppFMOD_DC           = 17,
  ppSH_MIX            = 18,
  ppSH_RUNGLER1       = 19,
  ppSH_RUNGLER2       = 20,
  ppLADDER1_FREQ      = 21,
  ppLADDER2_FREQ      = 22,
  ppLADDER_Q          = 23,
  ppSTEREO            = 24,
  // multibuttons selected
  ppJAM_MB_OSC1           = 25,
  ppJAM_MB_OSC2           = 26,
  ppJAM_MB_BJ_MIX         = 27,
  ppJAM_MB_BJ_FILTER      = 28,
  ppJAM_MB_BLIPPO_MIX     = 29,
  ppJAM_MB_BLIPPO_FILTER  = 30
};
#define PATCH_SIZE 31

const byte cc_map[] = // aligns with patch elements
{
  0,            // mode is effected by program change, not a cc
  ccVOLUME,
  ccOSC1_FREQ,
  ccOSC1_SH_MOD,
  ccOSC1_RUNGLER_MOD,
  ccOSC1_CROSS_MOD,
  ccOSC2_FREQ,
  ccOSC2_RUNGLER1_MOD,
  ccOSC2_RUNGLER2_MOD,
  ccOSC2_CROSS_MOD,
  ccCOMPARE_MIX,
  ccOSC1_MIX,
  ccOSC2_MIX,
  ccFILTER_FREQ,
  ccFILTER_Q,
  ccFMOD_RUNGLER1,
  ccFMOD_OSC2,
  ccFMOD_DC,
  ccSH_MIX,
  ccSH_RUNGLER1,
  ccSH_RUNGLER2,
  ccLADDER1_FREQ,
  ccLADDER2_FREQ,
  ccLADDER_Q,
  ccSTEREO
};
#define CC_SIZE sizeof(cc_map)

struct ee_block
{
  // storage overhead/housekeeping
  byte polarity;
  // settings
  byte brightness;
  byte channel;
  // MIDI overridable
  byte patch[PATCH_SIZE];
};

class EEdatabase : public ExternalEEPROM
{
  public:
    ee_block eeprom_data; // copy of EEPROM
    ee_block ui_data;     // current UI settings
    ee_block ui_disp;     // displayed UI settings
    ee_block live_data;   // what teensy knows
    
    uint settings_index;  // EEPROM block index of current settings

    boolean dial_changed;
    
  //-- load database from EEPROM
  
    void loadDatabase() // to eeprom_data
    { 
      // EEPROM on I2C
      Wire.begin();
      Wire.setClock(400000);
      // all this does is check the I2C bus to see if anything's there
      if (!begin())
        fatalErrorHandler(BLINK_ERR_EEPROM,"NO EEPROM"); // never returns
      // we have to tell it the chip size
      //   24C256 = 32K
      setMemorySize(32768);
      // check for initialized EEPROM
      { uint16_t s;
        read( EE_SIGNATURE_ADDR, (byte *)&s, 2 );
DEBUG_VALUE("signature ",s);  
        if (s != EE_SIGNATURE) { initializeEEPROM(); return; }
      }
      // load current settings
      //   locate current block (search for polarity change)  
      read( 0, (uint8_t *)&eeprom_data, 1 ); // read 1'st polarity byte
      settings_index = EE_DATABLOCKS-1; // current setting is last block if all polarity is same
      for (uint i=1; i < EE_DATABLOCKS; i++)
      { uint8_t pol;
        read( i*64, &pol, 1 );
        if (eeprom_data.polarity != pol) { settings_index = i-1;  break; }
      }
DEBUG_VALUE("settings @ ",settings_index);  
      // load remainder of data from block
      read( settings_index*64+1, (uint8_t *)&eeprom_data+1, sizeof(eeprom_data)-1 );
      // initialize state
      dial_changed = false;
      // force valid data
      if (eeprom_data.patch[ppMODE]                 > 3) eeprom_data.patch[ppMODE]                 = 0;
      if (eeprom_data.patch[ppJAM_MB_OSC1]          > 2) eeprom_data.patch[ppJAM_MB_OSC1]          = 0;
      if (eeprom_data.patch[ppJAM_MB_OSC2]          > 2) eeprom_data.patch[ppJAM_MB_OSC2]          = 0;
      if (eeprom_data.patch[ppJAM_MB_BJ_MIX]        > 2) eeprom_data.patch[ppJAM_MB_BJ_MIX]        = 0;
      if (eeprom_data.patch[ppJAM_MB_BJ_FILTER]     > 4) eeprom_data.patch[ppJAM_MB_BJ_FILTER]     = 0;
      if (eeprom_data.patch[ppJAM_MB_BLIPPO_MIX]    > 2) eeprom_data.patch[ppJAM_MB_BLIPPO_MIX]    = 0;
      if (eeprom_data.patch[ppJAM_MB_BLIPPO_FILTER] > 2) eeprom_data.patch[ppJAM_MB_BLIPPO_FILTER] = 0;
      for (uint i=0; i < CC_SIZE; i++) eeprom_data.patch[i] &= 0x7F;
    }

  //-- initialize new EEPROM

    void initializeEEPROM() // to eeprom_data
    {
DEBUG_MESSAGE("Initializing EEPROM...");
      // set signature
      { uint16_t s = EE_SIGNATURE;
        write( EE_SIGNATURE_ADDR, (byte *)&s, 2 );
      }      
      // reset polarity bit on all blcoks
      { byte p = 0xFF;
        for (uint i=0; i < EE_DATABLOCKS; i++)
          write( 64*i, (byte *)&p, 1 );
      }
      // initialize database
      for (uint i=0; i < PATCH_SIZE; i++) eeprom_data.patch[i] = 0;
      // write 1'st block
      eeprom_data.polarity   = 0;
      eeprom_data.brightness = 127;
      eeprom_data.channel    = 13;
      write( 0, (byte *)&eeprom_data, sizeof(eeprom_data) );
    }
   
  //-- save the current settings to EEPROM in load leveled memory
  
    void saveSettings() // store eeprom_data to wear leveled EE
    {
      // don't write if same data   
      ee_block orig;
      read( settings_index*64, (uint8_t*)&orig, sizeof(ee_block) );
      if (memcmp( (uint8_t*)&orig, (uint8_t*)&eeprom_data, sizeof(eeprom_data) ) == 0) return;   
DEBUG_MESSAGE("writing EE");      
      // bump to next block
      settings_index = (settings_index+1) % EE_DATABLOCKS;
DEBUG_VALUE("settings # ",settings_index);    
      // flip polarity if wrap
      if (settings_index == 0) eeprom_data.polarity ^= 1;
      // update EE
      write( settings_index*64, (uint8_t*)&eeprom_data, sizeof(eeprom_data) );  
    }
 
  //-- detect changes and save them to EEPROM
  //   save is delayed while changes continue to be made (EEPROM write flood prevention)
  //   MIDI-only changes are not autosaved

    void autoSave() // may block for ~ 5-10 msec
    { static ee_block save;  // last values checked
      static uint32_t tm;    // autosave timer
      static uint state = 0; // 0=no changes, 1=watching changes
      if (!dial_changed) return; // only autosave after manual change
      switch (state)
      { case 0: // watch for initial change
          if (memcmp( (byte *)&ui_data, (byte *)&eeprom_data, sizeof(eeprom_data) ))
          { memcpy( (byte *)&save, (byte *)&ui_data, sizeof(eeprom_data) );
            state = 1;   
            tm = millis();
          }
          break;
        case 1: // watch for further changes, extending save delay
          if (memcmp( (byte *)&ui_data, (byte *)&eeprom_data, sizeof(eeprom_data) ) == 0)
            state = 0; // back to original settings
          else if (memcmp( (byte *)&ui_data, (byte *)&save, sizeof(eeprom_data) )) // further changes?
          { memcpy( (byte *)&save, (byte *)&ui_data, sizeof(eeprom_data) );
            tm = millis();
          }        
          else if (millis() - tm > AUTOSAVE_MSEC) // no changes for a while?
          { // update EEPROM
DEBUG_MESSAGE("EEPROM: AUTOSAVE");            
            memcpy( (byte *)&eeprom_data, (byte *)&ui_data, sizeof(eeprom_data) );
            saveSettings();            
            state = 0;
            dial_changed = false;
          }
      }
    }
};

#endif
