//------------------------------------------------------
// screen displays

#include "app.h"
#include "common.h"
#include "pages.h"
#include "help.h"

// title font
#include "font_Sat_32.h" // http://oleddisplay.squix.ch/#/home
#define CF_S32 &Satisfy_Regular_32
// help page body text font
#include "font_DVu_10.h"
#define CF_DV10 &DejaVu_LGC_Sans_Condensed_10

// in main...
extern EEdatabase DB;
extern TFT_eSPI tft;
extern PageController page_controller;

// splash screen timeout
#define SPLASH_MSEC 10000

// display geometry
#define X_MARGIN   5 // horizontal margin
#define TITLE_Y    5 // title
#define ROW1_Y    40 // top knob row
#define ROW2_Y   140 // middle knob row
#define ROW3_Y   250 // bottom knob row
#define LINESP_9  17 // line spacing
#define LINESP_12 22 // line spacing
#define BAR_H     25 // bar widget
#define BAR_W    100 // progress bar width
#define DBAR_H    20 // double bar height
#define DBAR_W   100 // double bar width
#define BACKGND_COLOR TFT_BLACK
#define TITLE_COLOR   TFT_ORANGE
#define MIDI_COLOR    TFT_SKYBLUE // midi values
#define TEXT_COLOR    TFT_WHITE
#define DATA_COLOR    TFT_YELLOW  // user-changable values
#define BAR_BORDER_COLOR DATA_COLOR
#define DARKDARKGRAY tft.color565(64,64,64)
#define HELP_COLOR    TFT_WHITE

String MODE_DESC[NUM_MODES] = { "Benjolin", "Blippo 1", "Blippo 2", "Blippo 3" };

//------------------------------------------------------
// fatal error handler

void fatalErrorHandler( uint blinks, String msg ) { page_controller.fatalError(blinks,msg); }

void PageController::fatalError( uint blinks, String msg )
{ // show error message on screen
  tft.fillScreen(BACKGND_COLOR);
  tft.setTextDatum(TL_DATUM); 
  tft.setFreeFont(FF17);             
  tft.drawString("ERROR", 5,5, GFXFF);
  tft.setFreeFont(FF18);             
  tft.drawString(msg, 5,40, GFXFF);
  // blink code on LED
  do // forever
  { digitalWrite( STATUS_LED_PIN, LOW );
    delay(1000);
    for (uint i=0; i < blinks; i++)
    { digitalWrite( STATUS_LED_PIN, HIGH );
      delay(250);
      digitalWrite( STATUS_LED_PIN, LOW );
      delay(250);
    }
  } while(1);
}

//------------------------------------------------------
// Display helpers

// paint a progress-bar style level indicator
void paintBar( uint x, uint y, uint v, uint maxv = 127 )
{
  if (v > maxv) v = maxv;
  tft.drawRect( x,y, BAR_W,BAR_H, BAR_BORDER_COLOR);
  tft.fillRect( x+1,y+1, BAR_W-2,BAR_H-2, tft.color565(64,64,64) ); //TFT_DARKGREY
  uint w = map( v, 0,maxv, 0, BAR_W-2 );
  if (w) tft.fillRect( x+1,y+1, w,BAR_H-2, TFT_GREEN );
}
  
// paint midi numeric value
void paintMidiValue( uint cc, uint midi_value, uint r_just, uint y, uint bg_color = BACKGND_COLOR )
{
  if (!page_controller.show_midi_values) return;
  tft.setTextDatum(r_just ? TR_DATUM:TL_DATUM);
  tft.setTextColor( MIDI_COLOR, bg_color );
  tft.setFreeFont(FSS9);      
  uint x = r_just ? DISP_W-X_MARGIN : X_MARGIN;
  String s = String(cc) + ":" + String(midi_value);
  tft.setTextPadding( tft.textWidth("999:999 ") );
  tft.drawString( s, x,y );
  tft.setTextPadding(0);
}

//------------------------------------------------------
// Pages

// helper to update UI and repaint if live value has been changed (via pass-thru CC presumably)
#define UPDATE_DISP_FROM_UI(property,renderer_fcn) \
 if (DB.ui_disp.patch[property] != DB.ui_data.patch[property]) \
 { DB.ui_disp.patch[property] = DB.ui_data.patch[property]; renderer_fcn; DB.dial_changed=true; }
  
// flip a property value to zero and back
#define BUTTON_FLIP(property,save_var,flip_value,max_value,renderer_fcn) \
 { if (DB.ui_data.patch[property] != flip_value) \
      { save_var = DB.ui_data.patch[property];  DB.ui_data.patch[property] = DB.ui_disp.patch[property] = flip_value; } \
   else DB.ui_data.patch[property] = DB.ui_disp.patch[property] = (save_var == 0) ? max_value:save_var; \
   renderer_fcn; DB.dial_changed=true; }

// increment a selector property with wrap
#define BUTTON_BUMP(property,count,renderer_fcn) \
 { uint nvalue = (DB.ui_data.patch[property]+1) % count; \
   DB.ui_data.patch[property] = DB.ui_disp.patch[property] = nvalue; \
   renderer_fcn; DB.dial_changed=true; }

// increment/decrement a CC (0..127) value
#define SPIN_VALUE(property,renderer_fcn) \
 { int n = (int)DB.ui_data.patch[property] + spin; if (n > 127) n = 127; if (n < 0) n = 0; \
   DB.ui_data.patch[property] = DB.ui_disp.patch[property] = n; \
   renderer_fcn; DB.dial_changed=true; }

// increment/decrement a selector with wrap
#define SPIN_SELECTOR(property,count,renderer_fcn) \
 { spin = spin > 0 ? 1 : -1; \
   int n = ((int)DB.ui_data.patch[property] + spin) % count; n = (n+count) % count; \
   DB.ui_data.patch[property] = DB.ui_disp.patch[property] = n; \
   renderer_fcn; DB.dial_changed=true; }

// - - - - - - - - - - - - -
// Splash page
// - - - - - - - - - - - - -

void splashPage::init()
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setTextColor( TITLE_COLOR, BACKGND_COLOR );  
  tft.setTextDatum(TC_DATUM); // origin is top center
  tft.setFreeFont(CF_S32);
  tft.drawString( APP_NAME, DISP_W/2,90); 

  tft.setTextColor( TEXT_COLOR, BACKGND_COLOR );  
  tft.setFreeFont(FSS12);             
  tft.drawString( "Benjolin & Blippo", DISP_W/2,150); 
  tft.drawString( "Synthesizer", DISP_W/2,175); 

  tft.setFreeFont(FSS9);             
  tft.drawString( APP_DATE, DISP_W/2,270); 
  tft.drawString( "www.FearlessNight.com", DISP_W/2,290); 

  tm = millis();
}
KNOBINFO_T splashPage::knobInfo([[maybe_unused]]uint knob_index) { return { ROTOR_COLOR_BLK, VelociBus::ACCELERATION_NONE }; }
void splashPage::loop()
{
  if (millis() - tm > SPLASH_MSEC) page_controller.start( PG_HOME );
}
void splashPage::spinHandler([[maybe_unused]]int8_t address, [[maybe_unused]]int8_t spin)
{
  page_controller.start( PG_HOME );
}
void splashPage::pressHandler([[maybe_unused]]int8_t address, [[maybe_unused]]VelociBus::BUTTON_EVENT event)
{
  page_controller.start( PG_HOME );
}

// - - - - - - - - - - - - -
// Home page
// - - - - - - - - - - - - -

void HomePage::init()
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setTextColor( TITLE_COLOR, BACKGND_COLOR );
  tft.setTextDatum(TC_DATUM); // origin is top center
  tft.setFreeFont(CF_S32);
  tft.drawString( APP_NAME, DISP_W/2,TITLE_Y); 

  tft.setTextColor( TEXT_COLOR, BACKGND_COLOR );  
  tft.setFreeFont(FSS12);  
  tft.drawString( "Benjolin & Blippo", DISP_W/2,60); 
  tft.drawString( "Synthesizer", DISP_W/2,85); 

  tft.setFreeFont(FSS12);  
  tft.setTextDatum(TL_DATUM); // origin is top left
  tft.drawString("Mode",      X_MARGIN,ROW3_Y); 
  tft.setTextDatum(TR_DATUM); // origin is top right
  tft.drawString("Output",    DISP_W-X_MARGIN,ROW2_Y); 
  tft.drawString("Volume",    DISP_W-X_MARGIN,ROW3_Y); 
  for (uint i=0; i < PATCH_SIZE; i++) DB.ui_disp.patch[i] = 0x80; // refresh display
}
void HomePage::paintMode()
{
  byte v = DB.ui_data.patch[ppMODE];
  if (v >= NUM_MODES) v = 0;
  tft.setTextDatum(TL_DATUM);
  tft.setFreeFont(FSS12);
  tft.setTextColor( DATA_COLOR, BACKGND_COLOR );  
  tft.setTextPadding( tft.textWidth(MODE_DESC[0])+5 );
  tft.drawString( MODE_DESC[v], X_MARGIN, ROW3_Y+LINESP_12+5 );
  tft.setTextPadding(0);
}
void HomePage::paintVolume()
{
  uint v = DB.ui_data.patch[ppVOLUME];
  paintBar( DISP_W-X_MARGIN-BAR_W, ROW3_Y+LINESP_12, v );
  paintMidiValue( ccVOLUME, v, 1, ROW3_Y+LINESP_12+LINESP_12+5, BACKGND_COLOR );
}
void HomePage::paintStereo()
{
  byte v = DB.ui_data.patch[ppSTEREO];
  tft.setTextDatum(TR_DATUM);
  tft.setFreeFont(FSS12);
  tft.setTextColor( DATA_COLOR, BACKGND_COLOR );  
  tft.setTextPadding( tft.textWidth("Stereo") );
  tft.drawString( v ? "Stereo" : "Mono", DISP_W-X_MARGIN, ROW2_Y+LINESP_12+5 );
  tft.setTextPadding(0);
}

void HomePage::loop()
{
  // scan for patch property changes applicable to this page
  UPDATE_DISP_FROM_UI( ppMODE,   paintMode() );
  UPDATE_DISP_FROM_UI( ppVOLUME, paintVolume() );
  UPDATE_DISP_FROM_UI( ppSTEREO, paintStereo() );
}

KNOBINFO_T HomePage::knobInfo(uint knob_index)
{ const KNOBINFO_T my_info[6] = 
  { { COLOR_RED, VelociBus::ACCELERATION_NONE}, // UL
    { COLOR_RED, VelociBus::ACCELERATION_NONE}, // ML
    { COLOR_GRN, VelociBus::ACCELERATION_NONE}, // LL
    { COLOR_RED, VelociBus::ACCELERATION_NONE}, // UR
    { COLOR_GRN, VelociBus::ACCELERATION_NONE}, // MR
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }  // LR
  };
  if (knob_index < 6) return my_info[knob_index]; else return {0,VelociBus::ACCELERATION_NONE};
}

void HomePage::spinHandler(int8_t address, int8_t spin)
{
  switch (address)
  {
//  case KNOB_UL: EMPTY   
//  case KNOB_UR: EMPTY   
//  case KNOB_ML: EMPTY
    case KNOB_MR: SPIN_SELECTOR(ppSTEREO,       2, paintStereo()); break;
    case KNOB_LL: SPIN_SELECTOR(ppMODE, NUM_MODES, paintMode());   break;
    case KNOB_LR: SPIN_VALUE(ppVOLUME,             paintVolume());
  }
}
void HomePage::pressHandler(int8_t address, VelociBus::BUTTON_EVENT event)
{
  if (event == VelociBus::press)
    switch (address)
    {
  //  case KNOB_UL: EMPTY   
  //  case KNOB_UR: EMPTY   
  //  case KNOB_ML: EMPTY
      case KNOB_MR: BUTTON_BUMP(ppSTEREO, 2,            paintStereo()); break;
      case KNOB_LL: BUTTON_BUMP(ppMODE,   NUM_MODES,    paintMode());   break;
      case KNOB_LR: BUTTON_FLIP(ppVOLUME, volume,0,127, paintVolume());         
    }
  else if (event == VelociBus::hold)
    page_controller.help( PG_HOME, address-1 );
}

// - - - - - - - - - - - - -
// display helpers
// - - - - - - - - - - - - -

#define ROW1 0
#define ROW2 1
#define ROW3 2
#define COL1 0
#define COL2 1
uint ROW_Y[3] = { ROW1_Y, ROW2_Y, ROW3_Y };

void paintFrequency( uint r, uint c, byte v, float f, uint ccValue )
{
  String s;
  if      (f <    1.0) s = String(f,4);
  else if (f <   10.0) s = String(f,3);
  else if (f <  100.0) s = String(f,2);
  else if (f < 1000.0) s = String(f,1);
  else                 s = String(f,0);
  s += "Hz";
  tft.setTextColor( DATA_COLOR, BACKGND_COLOR );  
  tft.setTextDatum( (c == COL1) ? TL_DATUM : TR_DATUM);
  tft.setFreeFont(FSS12);
  uint y = ROW_Y[ r ] + LINESP_12+5;
  uint x = (c == COL1) ? X_MARGIN : (DISP_W-X_MARGIN);
  uint tw = tft.textWidth("9.9999HzX");
  tft.setTextPadding( tw );
  tft.drawString( s,x,y );
  tft.setTextPadding(0);
  // MIDI value
  paintMidiValue( ccValue, v, c, y+LINESP_12, BACKGND_COLOR );
}

void paintOSC_Frequency   ( uint r, uint c, uint ppIndex, uint ccValue ) { byte v = DB.ui_data.patch[ppIndex]; float f = OSC_FREQ(v);    paintFrequency( r,c,v,f,ccValue ); }
void paintFILTER_Frequency( uint r, uint c, uint ppIndex, uint ccValue ) { byte v = DB.ui_data.patch[ppIndex]; float f = FILTER_FREQ(v); paintFrequency( r,c,v,f,ccValue ); }
void paintLADDER_Frequency( uint r, uint c, uint ppIndex, uint ccValue ) { byte v = DB.ui_data.patch[ppIndex]; float f = LADDER_FREQ(v); paintFrequency( r,c,v,f,ccValue ); }

#define paintOSC1_SHmod(r,c) paintCommonBar(r,c,ppOSC1_SH_MOD,ccOSC1_SH_MOD)
#define paintOSC1_RungerMod(r,c) paintCommonBar(r,c,ppOSC1_RUNGLER_MOD,ccOSC1_RUNGLER_MOD)
#define paintOSC1_CrossMod(r,c) paintCommonBar(r,c,ppOSC1_CROSS_MOD,ccOSC1_CROSS_MOD)

#define paintOSC2_Runger1Mod(r,c) paintCommonBar(r,c,ppOSC2_RUNGLER1_MOD,ccOSC2_RUNGLER1_MOD)
#define paintOSC2_Runger2Mod(r,c) paintCommonBar(r,c,ppOSC2_RUNGLER2_MOD,ccOSC2_RUNGLER2_MOD)
#define paintOSC2_CrossMod(r,c) paintCommonBar(r,c,ppOSC2_CROSS_MOD,ccOSC2_CROSS_MOD)

#define paintBJ_Compare(r,c) paintCommonBar(r,c,ppCOMPARE_MIX,ccCOMPARE_MIX)
#define paintBJ_OSC1(r,c) paintCommonBar(r,c,ppOSC1_MIX,ccOSC1_MIX)
#define paintBJ_OSC2(r,c) paintCommonBar(r,c,ppOSC2_MIX,ccOSC2_MIX)

#define paintBJ_Resonance(r,c) paintCommonBar(r,c,ppFILTER_Q,ccFILTER_Q)
#define paintBJ_Runger(r,c) paintCommonBar(r,c,ppFMOD_RUNGLER1,ccFMOD_RUNGLER1)
#define paintBJ_OSCmix(r,c) paintCommonBar(r,c,ppFMOD_OSC2,ccFMOD_OSC2)
#define paintBJ_DC(r,c) paintCommonBar(r,c,ppFMOD_DC,ccFMOD_DC)

#define paintBL_SHmix(r,c) paintCommonBar(r,c,ppSH_MIX,ccSH_MIX)
#define paintBL_Rungler1(r,c) paintCommonBar(r,c,ppSH_RUNGLER1,ccSH_RUNGLER1)
#define paintBL_Rungler2(r,c) paintCommonBar(r,c,ppSH_RUNGLER2,ccSH_RUNGLER2)
#define paintBL_Resonance(r,c) paintCommonBar(r,c,ppLADDER_Q,ccLADDER_Q)

void paintCommonBar( uint r, uint c, uint ppIndex, uint ccValue )
{
  uint v = DB.ui_data.patch[ppIndex];
  uint x = (c == COL1) ? X_MARGIN : (DISP_W-X_MARGIN-BAR_W);
  uint y = ROW_Y[ r ] + LINESP_12;
  paintBar( x,y,v );
  paintMidiValue( ccValue, v, c, y+LINESP_12+5, BACKGND_COLOR );
}

// - - - - - - - - - - - - -
// OSC1
// - - - - - - - - - - - - -

void OSC1Page::init()
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setTextColor( TITLE_COLOR, BACKGND_COLOR );
  tft.setTextDatum(TC_DATUM); // origin is top center
  tft.setFreeFont(FSSB18);             
  tft.drawString( "Oscillator 1", DISP_W/2,TITLE_Y); 
  tft.setTextColor( TEXT_COLOR, BACKGND_COLOR );  
  tft.setFreeFont(FSS12);  
  tft.setTextDatum(TL_DATUM); // origin is top left
  tft.drawString("Frequency",   X_MARGIN,ROW1_Y); 
  tft.setTextDatum(TR_DATUM); // origin is top right
  tft.drawString("S/H Mod",     DISP_W-X_MARGIN,ROW1_Y); 
  tft.drawString("Rungler Mod", DISP_W-X_MARGIN,ROW2_Y); 
  tft.drawString("Cross Mod",   DISP_W-X_MARGIN,ROW3_Y); 
  for (uint i=0; i < PATCH_SIZE; i++) DB.ui_disp.patch[i] = 0x80; // refresh display
}

void OSC1Page::loop()
{
  // scan for patch property changes applicable to this page
  UPDATE_DISP_FROM_UI( ppOSC1_FREQ,        paintOSC_Frequency (ROW1,COL1,ppOSC1_FREQ,ccOSC1_FREQ) );
  UPDATE_DISP_FROM_UI( ppOSC1_SH_MOD,      paintOSC1_SHmod    (ROW1,COL2) );
  UPDATE_DISP_FROM_UI( ppOSC1_RUNGLER_MOD, paintOSC1_RungerMod(ROW2,COL2) );
  UPDATE_DISP_FROM_UI( ppOSC1_CROSS_MOD,   paintOSC1_CrossMod (ROW3,COL2) );
}

KNOBINFO_T OSC1Page::knobInfo(uint knob_index)
{ const KNOBINFO_T my_info[6] = 
  { { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // UL
    { COLOR_RED, VelociBus::ACCELERATION_NONE}, // ML
    { COLOR_RED, VelociBus::ACCELERATION_NONE}, // LL
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // UR
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // MR
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }  // LR
  };
  if (knob_index < 6) return my_info[knob_index]; else return {0,VelociBus::ACCELERATION_NONE};
}

void OSC1Page::spinHandler(int8_t address, int8_t spin)
{
  switch (address)
  {
    case KNOB_UL: SPIN_VALUE(ppOSC1_FREQ,        paintOSC_Frequency (ROW1,COL1,ppOSC1_FREQ,ccOSC1_FREQ));   break;  
    case KNOB_UR: SPIN_VALUE(ppOSC1_SH_MOD,      paintOSC1_SHmod    (ROW1,COL2));   break;  
//  case KNOB_ML: EMPTY
    case KNOB_MR: SPIN_VALUE(ppOSC1_RUNGLER_MOD, paintOSC1_RungerMod(ROW2,COL2));   break;    
//  case KNOB_LL: EMPTY
    case KNOB_LR: SPIN_VALUE(ppOSC1_CROSS_MOD,   paintOSC1_CrossMod (ROW3,COL2));
  }
}
void OSC1Page::pressHandler(int8_t address, VelociBus::BUTTON_EVENT event)
{
  if (event == VelociBus::press)
    switch (address)
    {
      case KNOB_UL: BUTTON_FLIP(ppOSC1_FREQ,        freq,0,127, paintOSC_Frequency (ROW1,COL1,ppOSC1_FREQ,ccOSC1_FREQ));  break;
      case KNOB_UR: BUTTON_FLIP(ppOSC1_SH_MOD,     shmod,0,127, paintOSC1_SHmod    (ROW1,COL2));  break; 
  //  case KNOB_ML: EMPTY
      case KNOB_MR: BUTTON_FLIP(ppOSC1_RUNGLER_MOD, rung,0,127, paintOSC1_RungerMod(ROW2,COL2));  break;
  //  case KNOB_LL: EMPTY
      case KNOB_LR: BUTTON_FLIP(ppOSC1_CROSS_MOD,   osc1,0,127, paintOSC1_CrossMod (ROW3,COL2));         
    }
  else if (event == VelociBus::hold)
    page_controller.help( PG_OSC1, address-1 );
}
  
// - - - - - - - - - - - - -
// OSC2
// - - - - - - - - - - - - -

void OSC2Page::init()
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setTextColor( TITLE_COLOR, BACKGND_COLOR );
  tft.setTextDatum(TC_DATUM); // origin is top center
  tft.setFreeFont(FSSB18);             
  tft.drawString( "Oscillator 2", DISP_W/2,TITLE_Y); 
  tft.setTextColor( TEXT_COLOR, BACKGND_COLOR );  
  tft.setFreeFont(FSS12);  
  tft.setTextDatum(TL_DATUM); // origin is top left
  tft.drawString("Frequency",   X_MARGIN,ROW1_Y); 
  tft.setTextDatum(TR_DATUM); // origin is top right
  tft.drawString("Rungler 1",   DISP_W-X_MARGIN,ROW1_Y); 
  tft.drawString("Rungler 2",   DISP_W-X_MARGIN,ROW2_Y); 
  tft.drawString("Cross Mod",   DISP_W-X_MARGIN,ROW3_Y); 
  for (uint i=0; i < PATCH_SIZE; i++) DB.ui_disp.patch[i] = 0x80; // refresh display
}

void OSC2Page::loop()
{
  // scan for patch property changes applicable to this page
  UPDATE_DISP_FROM_UI( ppOSC2_FREQ,         paintOSC_Frequency(ROW1,COL1,ppOSC2_FREQ,ccOSC2_FREQ) );
  UPDATE_DISP_FROM_UI( ppOSC2_RUNGLER1_MOD, paintOSC2_Runger1Mod(ROW1,COL2) );
  UPDATE_DISP_FROM_UI( ppOSC2_RUNGLER2_MOD, paintOSC2_Runger2Mod(ROW2,COL2) );
  UPDATE_DISP_FROM_UI( ppOSC2_CROSS_MOD,    paintOSC2_CrossMod  (ROW3,COL2) );
}

KNOBINFO_T OSC2Page::knobInfo(uint knob_index)
{ const KNOBINFO_T my_info[6] = 
  { { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // UL
    { COLOR_RED, VelociBus::ACCELERATION_NONE}, // ML
    { COLOR_RED, VelociBus::ACCELERATION_NONE}, // LL
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // UR
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // MR
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }  // LR
  };
  if (knob_index < 6) return my_info[knob_index]; else return {0,VelociBus::ACCELERATION_NONE};
}

void OSC2Page::spinHandler(int8_t address, int8_t spin)
{
  switch (address)
  {
    case KNOB_UL: SPIN_VALUE(ppOSC2_FREQ,         paintOSC_Frequency  (ROW1,COL1,ppOSC2_FREQ,ccOSC2_FREQ));   break;  
    case KNOB_UR: SPIN_VALUE(ppOSC2_RUNGLER1_MOD, paintOSC2_Runger1Mod(ROW1,COL2));   break;  
//  case KNOB_ML: EMPTY
    case KNOB_MR: SPIN_VALUE(ppOSC2_RUNGLER2_MOD, paintOSC2_Runger2Mod(ROW2,COL2));   break;    
//  case KNOB_LL: EMPTY
    case KNOB_LR: SPIN_VALUE(ppOSC2_CROSS_MOD,    paintOSC2_CrossMod  (ROW3,COL2));
  }
}
void OSC2Page::pressHandler(int8_t address, VelociBus::BUTTON_EVENT event)
{
  if (event == VelociBus::press)
    switch (address)
    {
      case KNOB_UL: BUTTON_FLIP(ppOSC2_FREQ,         freq, 0,127, paintOSC_Frequency  (ROW1,COL1,ppOSC2_FREQ,ccOSC2_FREQ));  break;
      case KNOB_UR: BUTTON_FLIP(ppOSC2_RUNGLER1_MOD, rung1,0,127, paintOSC2_Runger1Mod(ROW1,COL2));  break; 
  //  case KNOB_ML: EMPTY
      case KNOB_MR: BUTTON_FLIP(ppOSC2_RUNGLER2_MOD, rung2,0,127, paintOSC2_Runger2Mod(ROW2,COL2));  break;
  //  case KNOB_LL: EMPTY
      case KNOB_LR: BUTTON_FLIP(ppOSC2_CROSS_MOD,    cross,0,127, paintOSC2_CrossMod  (ROW3,COL2));         
    }
  else if (event == VelociBus::hold)
    page_controller.help( PG_OSC2, address-1 );
}

// - - - - - - - - - - - - -
// Benjolin Page 1 - Mix
// - - - - - - - - - - - - -

void BJ_MixPage::init()
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setTextColor( TITLE_COLOR, BACKGND_COLOR );
  tft.setTextDatum(TC_DATUM); // origin is top center
  tft.setFreeFont(FSSB18);             
  tft.drawString( "Benjolin Mix", DISP_W/2,TITLE_Y); 
  tft.setTextColor( TEXT_COLOR, BACKGND_COLOR );  
  tft.setFreeFont(FSS12);
  tft.setTextDatum(TL_DATUM); // origin is top left
  tft.drawString("Oscillator 1", X_MARGIN,ROW1_Y); 
  tft.drawString("Oscillator 2", X_MARGIN,ROW2_Y); 
  tft.drawString("Compare",      X_MARGIN,ROW3_Y); 
  for (uint i=0; i < PATCH_SIZE; i++) DB.ui_disp.patch[i] = 0x80; // refresh display
}

void BJ_MixPage::loop()
{
  // scan for patch property changes applicable to this page
  UPDATE_DISP_FROM_UI( ppOSC1_MIX,    paintBJ_OSC1   (ROW1,COL1) );
  UPDATE_DISP_FROM_UI( ppOSC2_MIX,    paintBJ_OSC2   (ROW2,COL1) );
  UPDATE_DISP_FROM_UI( ppCOMPARE_MIX, paintBJ_Compare(ROW3,COL1) );
}

KNOBINFO_T BJ_MixPage::knobInfo(uint knob_index)
{ const KNOBINFO_T my_info[6] = 
  { { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // UL
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // ML
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // LL
    { COLOR_RED, VelociBus::ACCELERATION_NONE}, // UR
    { COLOR_RED, VelociBus::ACCELERATION_NONE}, // MR
    { COLOR_RED, VelociBus::ACCELERATION_NONE}  // LR
  };
  if (knob_index < 6) return my_info[knob_index]; else return {0,VelociBus::ACCELERATION_NONE};
}

void BJ_MixPage::spinHandler(int8_t address, int8_t spin)
{
  switch (address)
  {
    case KNOB_UL: SPIN_VALUE(ppOSC1_MIX,    paintBJ_OSC1   (ROW1,COL1));   break;  
    case KNOB_ML: SPIN_VALUE(ppOSC2_MIX,    paintBJ_OSC2   (ROW2,COL1));   break; 
    case KNOB_LL: SPIN_VALUE(ppCOMPARE_MIX, paintBJ_Compare(ROW3,COL1));   break;
//  case KNOB_UR: EMPTY
//  case KNOB_MR: EMPTY    
//  case KNOB_LR: EMPTY
  }
}

void BJ_MixPage::pressHandler(int8_t address, VelociBus::BUTTON_EVENT event)
{
  if (event == VelociBus::press)
    switch (address)
    {
      case KNOB_UL: BUTTON_FLIP(ppOSC1_MIX, mix1, 0,127, paintBJ_OSC1(ROW1,COL1));  break;
      case KNOB_ML: BUTTON_FLIP(ppOSC2_MIX, mix2, 0,127, paintBJ_OSC2(ROW2,COL1));  break;
      case KNOB_LL: BUTTON_FLIP(ppCOMPARE_MIX, compare,0,127, paintBJ_Compare(ROW3,COL1));  break;
//    case KNOB_UR: EMPTY 
//    case KNOB_MR: EMPTY
//    case KNOB_LR: EMPTY         
    }
  else if (event == VelociBus::hold)
    page_controller.help( PG_BJ_MIX, address-1 );  
}
 
// - - - - - - - - - - - - -
// Benjolin Page 2 - Filter
// - - - - - - - - - - - - -

void BJ_FilterPage::init()
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setTextColor( TITLE_COLOR, BACKGND_COLOR );
  tft.setTextDatum(TC_DATUM); // origin is top center
  tft.setFreeFont(FSSB18);             
  tft.drawString( "Benjolin Filter", DISP_W/2,TITLE_Y); 
  tft.setTextColor( TEXT_COLOR, BACKGND_COLOR );  
  tft.setFreeFont(FSS12);
  tft.setTextDatum(TL_DATUM); // origin is top left
  tft.drawString("Frequency", X_MARGIN,ROW1_Y); 
  tft.drawString("Resonance", X_MARGIN,ROW2_Y); 
  tft.setTextDatum(TR_DATUM); // origin is top right
  tft.drawString("Rungler", DISP_W-X_MARGIN,ROW1_Y); 
  tft.drawString("Osc 2",   DISP_W-X_MARGIN,ROW2_Y); 
  tft.drawString("Fixed",   DISP_W-X_MARGIN,ROW3_Y); 
  for (uint i=0; i < PATCH_SIZE; i++) DB.ui_disp.patch[i] = 0x80; // refresh display
}

void BJ_FilterPage::loop()
{
  // scan for patch property changes applicable to this page
  UPDATE_DISP_FROM_UI( ppFILTER_FREQ,   paintFILTER_Frequency(ROW1,COL1,ppFILTER_FREQ,ccFILTER_FREQ) );
  UPDATE_DISP_FROM_UI( ppFILTER_Q,      paintBJ_Resonance (ROW2,COL1) );
  UPDATE_DISP_FROM_UI( ppFMOD_RUNGLER1, paintBJ_Runger    (ROW1,COL2) );
  UPDATE_DISP_FROM_UI( ppFMOD_OSC2,     paintBJ_OSCmix    (ROW2,COL2) );
  UPDATE_DISP_FROM_UI( ppFMOD_DC,       paintBJ_DC        (ROW3,COL2) );
}

KNOBINFO_T BJ_FilterPage::knobInfo(uint knob_index)
{ const KNOBINFO_T my_info[6] = 
  { { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // UL
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // ML
    { COLOR_RED, VelociBus::ACCELERATION_NONE}, // LL
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // UR
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // MR
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }  // LR
  };
  if (knob_index < 6) return my_info[knob_index]; else return {0,VelociBus::ACCELERATION_NONE};
}

void BJ_FilterPage::spinHandler(int8_t address, int8_t spin)
{
  switch (address)
  {
    case KNOB_UL: SPIN_VALUE(ppFILTER_FREQ,   paintFILTER_Frequency(ROW1,COL1,ppFILTER_FREQ,ccFILTER_FREQ)); break;  
    case KNOB_ML: SPIN_VALUE(ppFILTER_Q,      paintBJ_Resonance (ROW2,COL1));   break; 
//  case KNOB_LL: EMPTY
    case KNOB_UR: SPIN_VALUE(ppFMOD_RUNGLER1, paintBJ_Runger    (ROW1,COL2));   break;  
    case KNOB_MR: SPIN_VALUE(ppFMOD_OSC2,     paintBJ_OSCmix    (ROW2,COL2));   break;    
    case KNOB_LR: SPIN_VALUE(ppFMOD_DC,       paintBJ_DC        (ROW3,COL2));   break;  
  }
}

void BJ_FilterPage::pressHandler(int8_t address, VelociBus::BUTTON_EVENT event)
{
  if (event == VelociBus::press)
    switch (address)
    {
      case KNOB_UL: BUTTON_FLIP(ppFILTER_FREQ,   freq,0,127, paintFILTER_Frequency(ROW1,COL1,ppFILTER_FREQ,ccFILTER_FREQ));  break;
      case KNOB_ML: BUTTON_FLIP(ppFILTER_Q,         q,0,127, paintBJ_Resonance (ROW2,COL1));  break;
//    case KNOB_LL: EMPTY
      case KNOB_UR: BUTTON_FLIP(ppFMOD_RUNGLER1, rung,0,127, paintBJ_Runger    (ROW1,COL2));  break; 
      case KNOB_MR: BUTTON_FLIP(ppFMOD_OSC2,     osc2,0,127, paintBJ_OSCmix    (ROW2,COL2));  break;
      case KNOB_LR: BUTTON_FLIP(ppFMOD_DC,         dc,0,127, paintBJ_DC        (ROW3,COL2));  break;         
    }
  else if (event == VelociBus::hold)
    page_controller.help( PG_BJ_FILTER, address-1 ); 
}
     
// - - - - - - - - - - - - -
// Blippo Page
// - - - - - - - - - - - - -

void BlippoPage::init()
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setTextColor( TITLE_COLOR, BACKGND_COLOR );
  tft.setTextDatum(TC_DATUM); // origin is top center
  tft.setFreeFont(FSSB18);             
  tft.drawString( "Blippo", DISP_W/2,TITLE_Y); 
  tft.setTextColor( TEXT_COLOR, BACKGND_COLOR );  
  tft.setFreeFont(FSS12);
  tft.setTextDatum(TL_DATUM); // origin is top left
  tft.drawString("S/H Mix",     X_MARGIN,ROW1_Y); 
  tft.drawString("Rungler 1",   X_MARGIN,ROW2_Y); 
  tft.drawString("Rungler 2",   X_MARGIN,ROW3_Y); 
  tft.setTextDatum(TR_DATUM); // origin is top right
  tft.drawString("Filter 1",    DISP_W-X_MARGIN,ROW1_Y); 
  tft.drawString("Filter 2",    DISP_W-X_MARGIN,ROW2_Y); 
  tft.drawString("Filter Res",  DISP_W-X_MARGIN,ROW3_Y); 
  for (uint i=0; i < PATCH_SIZE; i++) DB.ui_disp.patch[i] = 0x80; // refresh display
}

void BlippoPage::loop()
{
  // scan for patch property changes applicable to this page
  UPDATE_DISP_FROM_UI( ppSH_MIX,        paintBL_SHmix         (ROW1,COL1) );
  UPDATE_DISP_FROM_UI( ppSH_RUNGLER1,   paintBL_Rungler1      (ROW2,COL1) );
  UPDATE_DISP_FROM_UI( ppSH_RUNGLER2,   paintBL_Rungler2      (ROW3,COL1) );
  UPDATE_DISP_FROM_UI( ppLADDER1_FREQ,  paintLADDER_Frequency (ROW1,COL2,ppLADDER1_FREQ,ccLADDER1_FREQ) );
  UPDATE_DISP_FROM_UI( ppLADDER2_FREQ,  paintLADDER_Frequency (ROW2,COL2,ppLADDER2_FREQ,ccLADDER2_FREQ) );
  UPDATE_DISP_FROM_UI( ppLADDER_Q,      paintBL_Resonance     (ROW3,COL2) );
}

KNOBINFO_T BlippoPage::knobInfo(uint knob_index)
{ const KNOBINFO_T my_info[6] = 
  { { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // UL
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // ML
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // LL
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // UR
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // MR
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }  // LR
  };
  if (knob_index < 6) return my_info[knob_index]; else return {0,VelociBus::ACCELERATION_NONE};
}

void BlippoPage::spinHandler(int8_t address, int8_t spin)
{
  switch (address)
  {
    case KNOB_UL: SPIN_VALUE(ppSH_MIX,       paintBL_SHmix         (ROW1,COL1));   break;  
    case KNOB_ML: SPIN_VALUE(ppSH_RUNGLER1,  paintBL_Rungler1      (ROW2,COL1));   break; 
    case KNOB_LL: SPIN_VALUE(ppSH_RUNGLER2,  paintBL_Rungler2      (ROW3,COL1));   break;  
    case KNOB_UR: SPIN_VALUE(ppLADDER1_FREQ, paintLADDER_Frequency (ROW1,COL2,ppLADDER1_FREQ,ccLADDER1_FREQ));   break;  
    case KNOB_MR: SPIN_VALUE(ppLADDER2_FREQ, paintLADDER_Frequency (ROW2,COL2,ppLADDER2_FREQ,ccLADDER2_FREQ));   break;      
    case KNOB_LR: SPIN_VALUE(ppLADDER_Q,     paintBL_Resonance     (ROW3,COL2));   break;  
  }
}

void BlippoPage::pressHandler(int8_t address, VelociBus::BUTTON_EVENT event)
{
  if (event == VelociBus::press)
    switch (address)
    {
      case KNOB_UL: BUTTON_FLIP(ppSH_MIX,       mix,     0,127, paintBL_SHmix         (ROW1,COL1));  break;
      case KNOB_ML: BUTTON_FLIP(ppSH_RUNGLER1,  rung1,   0,127, paintBL_Rungler1      (ROW2,COL1));  break;
      case KNOB_LL: BUTTON_FLIP(ppSH_RUNGLER2,  rung2,   0,127, paintBL_Rungler2      (ROW3,COL1));  break;
      case KNOB_UR: BUTTON_FLIP(ppLADDER1_FREQ, ladder1, 0,127, paintLADDER_Frequency (ROW1,COL2,ppLADDER1_FREQ,ccLADDER1_FREQ));  break; 
      case KNOB_MR: BUTTON_FLIP(ppLADDER2_FREQ, ladder2, 0,127, paintLADDER_Frequency (ROW2,COL2,ppLADDER2_FREQ,ccLADDER2_FREQ));  break;
      case KNOB_LR: BUTTON_FLIP(ppLADDER_Q,     q,       0,127, paintBL_Resonance     (ROW3,COL2));  break;         
    }
  else if (event == VelociBus::hold)
    page_controller.help( PG_BLIPPO, address-1 ); 
}
         
// - - - - - - - - - - - - -
// Jam Page
// - - - - - - - - - - - - -

void JamPage::caption( uint r, uint c, const GFXfont* f, String s1, String s2, uint pp )
{
  uint y = ROW_Y[ r ] - 22;
  uint x = (c == COL1) ? X_MARGIN : DISP_W/2;
  tft.fillRect( x,y, DISP_W/2-X_MARGIN,44, BACKGND_COLOR );
  // display caption
  tft.setFreeFont(FSS9);
  tft.setTextColor( TEXT_COLOR, BACKGND_COLOR );  
  tft.setFreeFont(f);
  tft.setTextDatum( (c == COL1) ? TL_DATUM:TR_DATUM);
  x = (c == COL1) ? X_MARGIN : (DISP_W-X_MARGIN);
  tft.drawString(s1,x,y);
  tft.drawString(s2,x,y+22);
  // force data to replot
  DB.ui_disp.patch[pp] = 0x80;  
}

#define captionOSC1SH()       caption(ROW2,COL1,FSS12,"Osc1 Mod", "S/H",      ppOSC1_SH_MOD)
#define captionOSC1Rungler()  caption(ROW2,COL1,FSS12,"Osc1 Mod", "Rungler",  ppOSC1_RUNGLER_MOD)
#define captionOSC1Cross()    caption(ROW2,COL1,FSS12,"Osc1 Mod", "Osc2",     ppOSC1_CROSS_MOD)
#define captionOSC2Rungler1() caption(ROW2,COL2,FSS12,"Osc2 Mod", "Rungler 1",ppOSC2_RUNGLER1_MOD)
#define captionOSC2Rungler2() caption(ROW2,COL2,FSS12,"Osc2 Mod", "Rungler 2",ppOSC2_RUNGLER2_MOD)
#define captionOSC2Cross()    caption(ROW2,COL2,FSS12,"Osc2 Mod", "Osc1",     ppOSC2_CROSS_MOD)

#define captionCompare()      caption(ROW3,COL1,FSS12,"Mix",        "Compare",  ppCOMPARE_MIX)
#define captionBJOSC1()       caption(ROW3,COL1,FSS12,"Mix",        "Osc 1",    ppOSC1_MIX)
#define captionBJOSC2()       caption(ROW3,COL1,FSS12,"Mix",        "Osc 2",    ppOSC2_MIX)
#define captionBJfilterFreq(){caption(ROW3,COL2,FSS12,"Filter",     "Frequency",ppFILTER_FREQ); tft.fillRect( DISP_W/2,ROW_Y[2]+22, DISP_W/2-X_MARGIN,BAR_H, BACKGND_COLOR ); }
#define captionBJfilterQ()    caption(ROW3,COL2,FSS12,"Filter",     "resonance",ppFILTER_Q)
#define captionBJrungler()    caption(ROW3,COL2,FSS12,"Filter Mod", "Rungler",  ppFMOD_RUNGLER1)
#define captionBJOSC2mod()    caption(ROW3,COL2,FSS12,"Filter Mod", "Osc 2",    ppFMOD_OSC2)
#define captionBJDC()         caption(ROW3,COL2,FSS12,"Filter Mod", "Fixed",    ppFMOD_DC)

#define captinBLmix()         caption(ROW3,COL1,FSS12,"Mix",      "S/H",      ppSH_MIX)
#define captionBLrungler1()   caption(ROW3,COL1,FSS12,"Mix",      "Rungler 1",ppSH_RUNGLER1)
#define captionBLrungler2()   caption(ROW3,COL1,FSS12,"Mix",      "Rungler 2",ppSH_RUNGLER2)
#define captionBLladder1()  { caption(ROW3,COL2,FSS12,"Ladder 1", "Frequency",ppLADDER1_FREQ); tft.fillRect( DISP_W/2,ROW_Y[2]+22, DISP_W/2-X_MARGIN,BAR_H, BACKGND_COLOR ); }
#define captionBLladder2()  { caption(ROW3,COL2,FSS12,"Ladder 2", "Frequency",ppLADDER2_FREQ); tft.fillRect( DISP_W/2,ROW_Y[2]+22, DISP_W/2-X_MARGIN,BAR_H, BACKGND_COLOR ); }
#define captionBLQ()          caption(ROW3,COL2,FSS12,"Ladder",   "resonance",ppLADDER_Q)

void JamPage::paintOSC1MB()
{
  switch (DB.ui_data.patch[ppJAM_MB_OSC1])
  { case  0: captionOSC1SH();       break;
    case  1: captionOSC1Rungler();  break;
    case  2: captionOSC1Cross();    break;
  }
}
void JamPage::paintOSC2MB()
{
  switch (DB.ui_data.patch[ppJAM_MB_OSC2])
  { case  0: captionOSC2Rungler1(); break;
    case  1: captionOSC2Rungler2(); break;
    case  2: captionOSC2Cross();    break;
  }
}
void JamPage::paintBJmixMB()
{ switch (DB.ui_data.patch[ppJAM_MB_BJ_MIX])
  { case  0: captionCompare(); break;
    case  1: captionBJOSC1();  break;
    case  2: captionBJOSC2();  break;
  }
}
void JamPage::paintBJfilterMB()
{
  switch (DB.ui_data.patch[ppJAM_MB_BJ_FILTER])
  { case  0: captionBJfilterFreq(); break;
    case  1: captionBJfilterQ();    break;
    case  2: captionBJrungler();    break;
    case  3: captionBJOSC2mod();    break;
    case  4: captionBJDC();         break;
  }
}
void JamPage::paintBLmixMB()
{ switch (DB.ui_data.patch[ppJAM_MB_BLIPPO_MIX])
  { case  0: captinBLmix();       break;
    case  1: captionBLrungler1(); break;
    case  2: captionBLrungler2(); break;
  }
}
void JamPage::paintBLfilterMB()
{
  switch (DB.ui_data.patch[ppJAM_MB_BLIPPO_FILTER])
  { case  0: captionBLladder1();  break;
    case  1: captionBLladder2();  break;
    case  2: captionBLQ();        break;
  }
}

void JamPage::init()
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setTextColor( TITLE_COLOR, BACKGND_COLOR );
  tft.setTextDatum(TC_DATUM); // origin is top center
  tft.setFreeFont(FSSB18);  
  String s = (DB.ui_data.patch[ppMODE] == MODE_BENJOLIN) ? "Benjolin Jam":"Blippo Jam";           
  tft.drawString( s, DISP_W/2,TITLE_Y); 
  tft.setTextColor( TEXT_COLOR, BACKGND_COLOR );  
  tft.setFreeFont(FSS12);
  tft.setTextDatum(TL_DATUM); // origin is top left
  tft.drawString("Osc 1",     X_MARGIN,ROW1_Y); 
  tft.setTextDatum(TR_DATUM); // origin is top right
  tft.drawString("Osc 2",    DISP_W-X_MARGIN,ROW1_Y); 
  // osc multibuttons
  paintOSC1MB();
  paintOSC2MB();
  // benjolin/blippo multibuttons
  if (DB.ui_data.patch[ppMODE] == MODE_BENJOLIN)
       { paintBJmixMB();  paintBJfilterMB(); }
  else { paintBLmixMB();  paintBLfilterMB(); }

  for (uint i=0; i < PATCH_SIZE; i++) DB.ui_disp.patch[i] = 0x80; // refresh display
}
  
void JamPage::loop()
{
  // scan for patch property changes applicable to this page
  UPDATE_DISP_FROM_UI( ppOSC1_FREQ, paintOSC_Frequency (ROW1,COL1,ppOSC1_FREQ,ccOSC1_FREQ) );
  UPDATE_DISP_FROM_UI( ppOSC2_FREQ, paintOSC_Frequency (ROW1,COL2,ppOSC2_FREQ,ccOSC2_FREQ) );
  
  if (DB.ui_data.patch[ppJAM_MB_OSC1]==0) UPDATE_DISP_FROM_UI( ppOSC1_SH_MOD,      paintOSC1_SHmod    (ROW2,COL1) );
  if (DB.ui_data.patch[ppJAM_MB_OSC1]==1) UPDATE_DISP_FROM_UI( ppOSC1_RUNGLER_MOD, paintOSC1_RungerMod(ROW2,COL1) );
  if (DB.ui_data.patch[ppJAM_MB_OSC1]==2) UPDATE_DISP_FROM_UI( ppOSC1_CROSS_MOD,   paintOSC1_CrossMod (ROW2,COL1) );

  if (DB.ui_data.patch[ppJAM_MB_OSC2]==0) UPDATE_DISP_FROM_UI( ppOSC2_RUNGLER1_MOD, paintOSC2_Runger1Mod(ROW2,COL2) );
  if (DB.ui_data.patch[ppJAM_MB_OSC2]==1) UPDATE_DISP_FROM_UI( ppOSC2_RUNGLER2_MOD, paintOSC2_Runger2Mod(ROW2,COL2) );
  if (DB.ui_data.patch[ppJAM_MB_OSC2]==2) UPDATE_DISP_FROM_UI( ppOSC2_CROSS_MOD,    paintOSC2_CrossMod  (ROW2,COL2) );

  if (DB.ui_data.patch[ppMODE] == MODE_BENJOLIN)
  {
    if (DB.ui_data.patch[ppJAM_MB_BJ_MIX]==0) UPDATE_DISP_FROM_UI( ppCOMPARE_MIX, paintBJ_Compare(ROW3,COL1) );
    if (DB.ui_data.patch[ppJAM_MB_BJ_MIX]==1) UPDATE_DISP_FROM_UI( ppOSC1_MIX,    paintBJ_OSC1   (ROW3,COL1) );
    if (DB.ui_data.patch[ppJAM_MB_BJ_MIX]==2) UPDATE_DISP_FROM_UI( ppOSC2_MIX,    paintBJ_OSC2   (ROW3,COL1) );
  
    if (DB.ui_data.patch[ppJAM_MB_BJ_FILTER]==0) UPDATE_DISP_FROM_UI( ppFILTER_FREQ,   paintFILTER_Frequency(ROW3,COL2,ppFILTER_FREQ,ccFILTER_FREQ) );
    if (DB.ui_data.patch[ppJAM_MB_BJ_FILTER]==1) UPDATE_DISP_FROM_UI( ppFILTER_Q,      paintBJ_Resonance    (ROW3,COL2) );
    if (DB.ui_data.patch[ppJAM_MB_BJ_FILTER]==2) UPDATE_DISP_FROM_UI( ppFMOD_RUNGLER1, paintBJ_Runger       (ROW3,COL2) );
    if (DB.ui_data.patch[ppJAM_MB_BJ_FILTER]==3) UPDATE_DISP_FROM_UI( ppFMOD_OSC2,     paintBJ_OSCmix       (ROW3,COL2) );
    if (DB.ui_data.patch[ppJAM_MB_BJ_FILTER]==4) UPDATE_DISP_FROM_UI( ppFMOD_DC,       paintBJ_DC           (ROW3,COL2) );
  }
  else // Blippo
  {
    if (DB.ui_data.patch[ppJAM_MB_BLIPPO_MIX]==0) UPDATE_DISP_FROM_UI( ppSH_MIX,        paintBL_SHmix   (ROW3,COL1) );
    if (DB.ui_data.patch[ppJAM_MB_BLIPPO_MIX]==1) UPDATE_DISP_FROM_UI( ppSH_RUNGLER1,   paintBL_Rungler1(ROW3,COL1) );
    if (DB.ui_data.patch[ppJAM_MB_BLIPPO_MIX]==2) UPDATE_DISP_FROM_UI( ppSH_RUNGLER2,   paintBL_Rungler2(ROW3,COL1) );

    if (DB.ui_data.patch[ppJAM_MB_BLIPPO_FILTER]==0) UPDATE_DISP_FROM_UI( ppLADDER1_FREQ,  paintLADDER_Frequency(ROW3,COL2,ppLADDER1_FREQ,ccLADDER1_FREQ) );
    if (DB.ui_data.patch[ppJAM_MB_BLIPPO_FILTER]==1) UPDATE_DISP_FROM_UI( ppLADDER2_FREQ,  paintLADDER_Frequency(ROW3,COL2,ppLADDER2_FREQ,ccLADDER2_FREQ) );
    if (DB.ui_data.patch[ppJAM_MB_BLIPPO_FILTER]==2) UPDATE_DISP_FROM_UI( ppLADDER_Q,      paintBL_Resonance    (ROW3,COL2) );
  }
}

KNOBINFO_T JamPage::knobInfo(uint knob_index)
{ const KNOBINFO_T my_info[6] = 
  { { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // UL
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // ML
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // LL
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // UR
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }, // MR
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }  // LR
  };
  if (knob_index < 6) return my_info[knob_index]; else return {0,VelociBus::ACCELERATION_NONE};
}

void JamPage::spinHandler(int8_t address, int8_t spin)
{
  switch (address)
  {
    case KNOB_UL: SPIN_VALUE(ppOSC1_FREQ, paintOSC_Frequency(ROW1,COL1,ppOSC1_FREQ,ccOSC1_FREQ)); break;  
    case KNOB_UR: SPIN_VALUE(ppOSC2_FREQ, paintOSC_Frequency(ROW1,COL2,ppOSC2_FREQ,ccOSC2_FREQ)); break; 
  // oscillator multibuttons    
    case KNOB_ML:  
      switch (DB.ui_data.patch[ppJAM_MB_OSC1])
      { case  0: SPIN_VALUE(ppOSC1_SH_MOD,      paintOSC1_SHmod    (ROW2,COL1));   break; 
        case  1: SPIN_VALUE(ppOSC1_RUNGLER_MOD, paintOSC1_RungerMod(ROW2,COL1));   break; 
        case  2: SPIN_VALUE(ppOSC1_CROSS_MOD,   paintOSC1_CrossMod (ROW2,COL1));   break; 
      }
      break;
    case KNOB_MR:
      switch (DB.ui_data.patch[ppJAM_MB_OSC2])
      { case  0: SPIN_VALUE(ppOSC2_RUNGLER1_MOD, paintOSC2_Runger1Mod(ROW2,COL2));   break; 
        case  1: SPIN_VALUE(ppOSC2_RUNGLER2_MOD, paintOSC2_Runger2Mod(ROW2,COL2));   break; 
        case  2: SPIN_VALUE(ppOSC2_CROSS_MOD,    paintOSC2_CrossMod  (ROW2,COL2));   break; 
      }
      break;
  // benjolin/blippo multibuttons
    case KNOB_LL:  
      if (DB.ui_data.patch[ppMODE] == MODE_BENJOLIN)
      { switch (DB.ui_data.patch[ppJAM_MB_BJ_MIX])
        { case  0: SPIN_VALUE(ppCOMPARE_MIX, paintBJ_Compare(ROW3,COL1));   break; 
          case  1: SPIN_VALUE(ppOSC1_MIX,    paintBJ_OSC1   (ROW3,COL1));   break; 
          case  2: SPIN_VALUE(ppOSC2_MIX,    paintBJ_OSC2   (ROW3,COL1));   break; 
        }
      }
      else // Blippo
      { switch (DB.ui_data.patch[ppJAM_MB_BLIPPO_MIX])
        { case  0: SPIN_VALUE(ppSH_MIX,      paintBL_SHmix   (ROW3,COL1));   break; 
          case  1: SPIN_VALUE(ppSH_RUNGLER1, paintBL_Rungler1(ROW3,COL1));   break; 
          case  2: SPIN_VALUE(ppSH_RUNGLER2, paintBL_Rungler2(ROW3,COL1));   break; 
        }
      }
      break;
    case KNOB_LR: 
      if (DB.ui_data.patch[ppMODE] == MODE_BENJOLIN)
      { switch (DB.ui_data.patch[ppJAM_MB_BJ_FILTER])
        { case  0: SPIN_VALUE(ppFILTER_FREQ,   paintFILTER_Frequency(ROW3,COL2,ppFILTER_FREQ,ccFILTER_FREQ)); break; 
          case  1: SPIN_VALUE(ppFILTER_Q,      paintBJ_Resonance    (ROW3,COL2));   break; 
          case  2: SPIN_VALUE(ppFMOD_RUNGLER1, paintBJ_Runger       (ROW3,COL2));   break; 
          case  3: SPIN_VALUE(ppFMOD_OSC2,     paintBJ_OSCmix       (ROW3,COL2));   break; 
          case  4: SPIN_VALUE(ppFMOD_DC,       paintBJ_DC           (ROW3,COL2));   break; 
        }
      }
      else // Blippo
      { switch (DB.ui_data.patch[ppJAM_MB_BLIPPO_FILTER])
        { case  0: SPIN_VALUE(ppLADDER1_FREQ, paintLADDER_Frequency(ROW3,COL2,ppLADDER1_FREQ,ccLADDER1_FREQ)); break; 
          case  1: SPIN_VALUE(ppLADDER2_FREQ, paintLADDER_Frequency(ROW3,COL2,ppLADDER2_FREQ,ccLADDER2_FREQ)); break; 
          case  2: SPIN_VALUE(ppLADDER_Q,     paintBJ_Compare      (ROW3,COL2));   break; 
        }
      }
  }
}

void JamPage::pressHandler(int8_t address, VelociBus::BUTTON_EVENT event)
{
  if (event == VelociBus::press)
    switch (address)
    {
      case KNOB_UL: BUTTON_FLIP(ppOSC1_FREQ, freq1,0,127, paintOSC_Frequency (ROW1,COL1,ppOSC1_FREQ,ccOSC1_FREQ));  break;
      case KNOB_UR: BUTTON_FLIP(ppOSC2_FREQ, freq2,0,127, paintOSC_Frequency (ROW1,COL2,ppOSC2_FREQ,ccOSC2_FREQ));  break;
    // multibuttons
      case KNOB_ML: DB.ui_data.patch[ppJAM_MB_OSC1] = (DB.ui_data.patch[ppJAM_MB_OSC1]+1) % 3;  paintOSC1MB();  break;
      case KNOB_MR: DB.ui_data.patch[ppJAM_MB_OSC2] = (DB.ui_data.patch[ppJAM_MB_OSC2]+1) % 3;  paintOSC2MB();  break; 
      case KNOB_LL: 
        if (DB.ui_data.patch[ppMODE] == MODE_BENJOLIN)
             { DB.ui_data.patch[ppJAM_MB_BJ_MIX]     = (DB.ui_data.patch[ppJAM_MB_BJ_MIX]+1) % 3;      paintBJmixMB(); }
        else { DB.ui_data.patch[ppJAM_MB_BLIPPO_MIX] = (DB.ui_data.patch[ppJAM_MB_BLIPPO_MIX]+1) % 3;  paintBLmixMB(); }
        break;      
      case KNOB_LR: 
        if (DB.ui_data.patch[ppMODE] == MODE_BENJOLIN)
             { DB.ui_data.patch[ppJAM_MB_BJ_FILTER]     = (DB.ui_data.patch[ppJAM_MB_BJ_FILTER]+1) % 5;      paintBJfilterMB(); }
        else { DB.ui_data.patch[ppJAM_MB_BLIPPO_FILTER] = (DB.ui_data.patch[ppJAM_MB_BLIPPO_FILTER]+1) % 3;  paintBLfilterMB(); }
    }
  else if (event == VelociBus::hold)
    page_controller.help( PG_JAM, address-1 );
}
 
// - - - - - - - - - - - - -
// Help
// - - - - - - - - - - - - -

void helpPage::init()
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setTextColor( TITLE_COLOR, BACKGND_COLOR );  
  tft.setTextDatum(TC_DATUM);
  tft.setFreeFont(FSS12);
  tft.drawString( help_database[parent].page_title, DISP_W/2,TITLE_Y);
  tft.setFreeFont(FSSB12);
  tft.drawString( help_database[parent].knob[button].title, DISP_W/2,TITLE_Y+25);

  tft.setTextDatum(TL_DATUM);
  tft.setFreeFont(CF_DV10);
  tft.setTextColor( HELP_COLOR, BACKGND_COLOR );  
  {
    uint y = 60;
    uint x = 0;
    uint i;
    uint i0 = 0;
    uint indent = 0;
    const uint line_spacing = 12;
    const uint char_spacing =  6;
    const uint indent_spacing = 10;
    //uint i = 0;
    for (i=0; i < help_database[parent].knob[button].body.length(); i++)
    { // scan to end of word
      switch (help_database[parent].knob[button].body.charAt(i))
      {
        case '\n':
        { 
          if (i > i0)
          {
            uint w = tft.textWidth(help_database[parent].knob[button].body.substring(i0,i));
            if (x+w > DISP_W-X_MARGIN-10) { y += line_spacing; x = 0; indent = indent_spacing; }
            tft.drawString( help_database[parent].knob[button].body.substring(i0,i), x+X_MARGIN+indent,y );
            x += w+char_spacing;
            i0 = i+1;
            if (x > 220) { y += line_spacing;  x = 0;  break; }
            // forced newline
            y += line_spacing;
          }
          else // half line space
            y += line_spacing/2;
          x = indent = 0;
          i0 = i+1;
          break;
        }
        case ' ':
        { uint w = tft.textWidth(help_database[parent].knob[button].body.substring(i0,i));
          if (x+w > DISP_W-X_MARGIN-10) { y += line_spacing; x = 0; indent = indent_spacing; }
          tft.drawString( help_database[parent].knob[button].body.substring(i0,i), x+X_MARGIN+indent,y );
          x += w+char_spacing;
          i0 = i+1;
        }
      }
    }
    if (i > i0)
    { uint w = tft.textWidth(help_database[parent].knob[button].body.substring(i0,i));
      if (x+w > DISP_W-X_MARGIN-10) { y += line_spacing; x = 0; indent = indent_spacing; }
      tft.drawString( help_database[parent].knob[button].body.substring(i0,i), x+X_MARGIN+indent,y );
    }
  }
}

KNOBINFO_T helpPage::knobInfo([[maybe_unused]]uint knob_index)
{
  return {COLOR_GRN,VelociBus::ACCELERATION_NONE};
}

void helpPage::pressHandler([[maybe_unused]]int8_t address, VelociBus::BUTTON_EVENT event)
{ 
  // any knob exits help
  if (event != VelociBus::press) return;
  page_controller.start( parent ); // return to base screen
}

void helpPage::spinHandler([[maybe_unused]]int8_t address, int8_t spin)
{
  // bump to adjacent help page
  int d = (spin > 0 ? 1 : -1);
  do // skip inactive knobs
  { int v = (int)button + d;
        v = (v+6) % 6;
    button = v;
  } while (help_database[parent].knob[button].title == "Help");
  init();
}

// - - - - - - - - - - - - -
// Screen Saver
// - - - - - - - - - - - - -

#define NUM_SAVER_COLORS 14
uint16_t colorset[NUM_SAVER_COLORS] = { TFT_BLUE,TFT_GREEN,TFT_CYAN,TFT_RED,TFT_MAGENTA,TFT_YELLOW,TFT_WHITE,TFT_ORANGE,TFT_GREENYELLOW,TFT_PINK,TFT_GOLD,TFT_SILVER,TFT_SKYBLUE,TFT_VIOLET};

void saverPage::init()
{
  // display the app name & current effect at a random location on the display
  tft.fillScreen(BACKGND_COLOR);
  tft.setTextColor( colorset[random(NUM_SAVER_COLORS)], BACKGND_COLOR );
  tft.setTextDatum(TC_DATUM); 
  tft.setFreeFont(FSSB18);
  uint w  = tft.textWidth( MODE_DESC[DB.ui_data.patch[ppMODE]] );
  uint h  = tft.fontHeight();
  tft.setFreeFont(FSS12);
  uint w2 = tft.textWidth(APP_NAME);
  uint h2 = tft.fontHeight();
  if (w2 > w) w = w2;  
  uint x = random( DISP_W - w ) + w/2;
  uint y = random( DISP_H - (h+h2) );
  tft.setFreeFont(FSSB18);
  tft.drawString( MODE_DESC[DB.ui_data.patch[ppMODE]], x,y );
  tft.setFreeFont(FSS12);
  tft.drawString( APP_NAME, x,y+h );
  tm = millis();
}

KNOBINFO_T saverPage::knobInfo([[maybe_unused]]uint knob_index)
{ 
  return {ROTOR_COLOR_BLK,VelociBus::ACCELERATION_NONE};
}

void saverPage::loop()
{
  if (millis() - tm >= SAVER_REPLOT_MSEC) init();
}

// - - - - - - - - - - - - -
// About
// - - - - - - - - - - - - -

void aboutPage::init()
{
  tft.fillScreen(BACKGND_COLOR);
  const uint org = 20;
  const uint box_color = TFT_BROWN;
  tft.fillRect( 10,10, DISP_W-20,210, box_color );
  tft.setTextColor( TEXT_COLOR, box_color );  
  tft.setTextDatum(TC_DATUM); // origin is top center
  tft.setFreeFont(CF_S32);
  tft.drawString( APP_NAME, DISP_W/2,org+3); 

  tft.setFreeFont(FSS12);             
  tft.drawString( "Benjolin & Blippo", DISP_W/2,org+57); 
  tft.drawString( "Synthesizer", DISP_W/2,org+80); 

  tft.setFreeFont(FSS9);             
  tft.drawString( APP_DATE, DISP_W/2,org+113); 
  tft.drawString( "Teensy4 & RP2040", 120,org+135); 
  tft.drawString( "www.FearlessNight.com", DISP_W/2,org+170); 
  // knobs
  tft.setTextColor( TEXT_COLOR, BACKGND_COLOR );  
  tft.setFreeFont(FSS9);      
  tft.setTextDatum(TC_DATUM); // origin is top center
  tft.setTextDatum(TL_DATUM); // origin is top left
  tft.drawString("Channel", X_MARGIN,ROW3_Y); 
  tft.setTextDatum(TR_DATUM); // origin is top right
  tft.drawString( "Brightness", DISP_W-X_MARGIN,ROW3_Y, GFXFF); 
  paintBrightness();
  paintChannel();
}

void aboutPage::paintBrightness()
{
  byte v = DB.ui_data.brightness;
  paintBar( DISP_W-X_MARGIN-BAR_W, ROW3_Y+LINESP_12, v, 255 );
}
void aboutPage::paintChannel()
{
  // 0 = omni 
  // 1..16 = channel  
  // 17 = off
  tft.setTextDatum(TL_DATUM);
  tft.setTextColor( DATA_COLOR, BACKGND_COLOR );  
  tft.setFreeFont(FSS12);
  tft.setTextPadding( tft.textWidth("OFF") );
  byte v = DB.ui_data.channel;
  String s;
  if (v == 0)      s = "All";
  else if (v < 16) s = String(v);
  else             s = "Off";
  tft.drawString( s, X_MARGIN+20, ROW3_Y+LINESP_12 );
  tft.setTextPadding(0);
}

KNOBINFO_T aboutPage::knobInfo(uint knob_index)
{ const KNOBINFO_T my_info[6] = 
  { { COLOR_RED, VelociBus::ACCELERATION_NONE}, // UL
    { COLOR_RED, VelociBus::ACCELERATION_NONE}, // ML
    { COLOR_GRN, VelociBus::ACCELERATION_NONE}, // LL
    { COLOR_RED, VelociBus::ACCELERATION_NONE}, // UR
    { COLOR_RED, VelociBus::ACCELERATION_NONE}, // MR
    { COLOR_GRN, VelociBus::ACCELERATION_MAX }  // LR
  };
  if (knob_index < 6) return my_info[knob_index]; else return {0,VelociBus::ACCELERATION_NONE};
}

// these work with the global properties...

#define GLOBAL_BUTTON_FLIP(property,save_var,flip_value,renderer_fcn) \
 { if (*property != flip_value) { save_var = *property;  *property = flip_value; } \
   else *property = save_var; \
   renderer_fcn; DB.dial_changed=true; }

#define GLOBAL_SPIN_VALUE(property,min_value,max_value,renderer_fcn) \
 { int n = (int)(*property) + spin; if (n > max_value) n = max_value; if (n < min_value) n = min_value; \
   *property = n; \
   renderer_fcn; DB.dial_changed=true; }

void aboutPage::loop()
{
  // scan for property changes applicable to this page
}

void aboutPage::spinHandler(int8_t address, int8_t spin)
{
  switch (address)
  {
    case KNOB_NAV: page_controller.start(PG_HOME); break;    
//  case KNOB_UL: EMPTY   
//  case KNOB_UR: EMPTY   
//  case KNOB_ML: EMPTY
//  case KNOB_MR: EMPTY     
    case KNOB_LL: GLOBAL_SPIN_VALUE(&DB.ui_data.channel, 0,17, paintChannel());   break;
    case KNOB_LR: GLOBAL_SPIN_VALUE(&DB.ui_data.brightness,MIN_BRIGHTNESS,MAX_BRIGHTNESS, paintBrightness());
                  page_controller.setBrightness(DB.ui_data.brightness);
  }
}

void aboutPage::pressHandler(int8_t address, VelociBus::BUTTON_EVENT event)
{
  if (event == VelociBus::press)
    switch (address)
    {
  //  case KNOB_UL: EMPTY   
  //  case KNOB_UR: EMPTY   
  //  case KNOB_ML: EMPTY
  //  case KNOB_MR: EMPTY     
      case KNOB_LL: GLOBAL_BUTTON_FLIP(&DB.ui_data.channel,    channel,0,      paintChannel());   break;   
      case KNOB_LR: GLOBAL_BUTTON_FLIP(&DB.ui_data.brightness, brightness,127, paintBrightness());         
                    page_controller.setBrightness(DB.ui_data.brightness);
    }
  else if (event == VelociBus::hold)
    page_controller.help( PG_ABOUT, address-1 );
}
