#pragma once
#ifndef Lib_Pages
#define Lib_Pages

#include <Arduino.h>
#include <vector>        // dynamic array

#include "app.h"
extern EEdatabase DB;
  
#include "FEARLESS_VelociBus.h"
extern VelociBus vbus;   // VelociBus interface
#define COLOR_RED 0b00001
#define COLOR_GRN 0b00100
#define COLOR_YEL 0b00101

struct KNOBINFO_T { uint color; VelociBus::ROTOR_ACCELERATION acceleration; };

#include "TFT_eSPI.h"
#include "Free_Fonts.h"

#define SCREENSAVE_MSEC (30*60000UL) // 30 minutes
#define SAVER_REPLOT_MSEC  (30000UL) // 30 seconds

//------------------------------------------------------
// Page base class

class Page
{
  public:
    virtual void loop() { ; }
    virtual void init() { ; }
    virtual void pressHandler([[maybe_unused]]int8_t address, [[maybe_unused]]VelociBus::BUTTON_EVENT event) { ; }
    virtual void spinHandler([[maybe_unused]]int8_t address, [[maybe_unused]]int8_t spin) { ; }
    virtual KNOBINFO_T knobInfo([[maybe_unused]]uint knob_index) { return { ROTOR_COLOR_RED, VelociBus::ACCELERATION_NONE }; }
};

//------------------------------------------------------
// Settings pages

// - - - - - - - - - - - - -
// About
class aboutPage : public Page
{
  public:
    void init();
    void loop();
    void pressHandler(int8_t address, VelociBus::BUTTON_EVENT event);  
    void spinHandler(int8_t address, int8_t spin);      
    void paintBrightness();
    void paintChannel();
    KNOBINFO_T knobInfo(uint knob_index);
  private:
    uint brightness;
    uint channel;
};

//------------------------------------------------------
// Synth control pages, dial selectable

// - - - - - - - - - - - - -
class HomePage : public Page
{
  public:
    void init();
    void loop();
    void spinHandler(int8_t address, int8_t spin);
    void pressHandler(int8_t address, VelociBus::BUTTON_EVENT event); 
    KNOBINFO_T knobInfo(uint knob_index);
  private:
    void paintMode();
    void paintVolume();
    void paintStereo();
  private:
    uint volume;
};

// - - - - - - - - - - - - -
class OSC1Page : public Page
{
  public:
    void init();
    void loop();
    void spinHandler(int8_t address, int8_t spin);
    void pressHandler(int8_t address, VelociBus::BUTTON_EVENT event); 
    KNOBINFO_T knobInfo(uint knob_index);
  private:
    uint freq,shmod,rung,osc1;
};

// - - - - - - - - - - - - -
class OSC2Page : public Page
{
  public:
    void init();
    void loop();
    void spinHandler(int8_t address, int8_t spin);
    void pressHandler(int8_t address, VelociBus::BUTTON_EVENT event); 
    KNOBINFO_T knobInfo(uint knob_index);
  private:
    uint freq,rung1,rung2,cross;
};

// - - - - - - - - - - - - -
class BJ_MixPage : public Page
{
  public:
    void init();
    void loop();
    void spinHandler(int8_t address, int8_t spin);
    void pressHandler(int8_t address, VelociBus::BUTTON_EVENT event); 
    KNOBINFO_T knobInfo(uint knob_index);
  private:
    uint mix1,mix2,compare;
};

// - - - - - - - - - - - - -
class BJ_FilterPage : public Page
{
  public:
    void init();
    void loop();
    void spinHandler(int8_t address, int8_t spin);
    void pressHandler(int8_t address, VelociBus::BUTTON_EVENT event); 
    KNOBINFO_T knobInfo(uint knob_index);
  private:
    uint freq,q,rung,osc2,dc;
};

// - - - - - - - - - - - - -
class BlippoPage : public Page
{
  public:
    void init();
    void loop();
    void spinHandler(int8_t address, int8_t spin);
    void pressHandler(int8_t address, VelociBus::BUTTON_EVENT event); 
    KNOBINFO_T knobInfo(uint knob_index);
  private:
    uint mix,rung1,rung2,ladder1,ladder2,q;
};

// - - - - - - - - - - - - -
class JamPage : public Page
{
  public:
    void init();
    void loop();
    void spinHandler(int8_t address, int8_t spin);
    void pressHandler(int8_t address, VelociBus::BUTTON_EVENT event); 
    KNOBINFO_T knobInfo(uint knob_index);
  private:
    uint freq1,freq2;
  private:
    void paintBLfilterMB();
    void paintBLmixMB();
    void paintBJfilterMB();
    void paintBJmixMB();
    void paintOSC2MB();
    void paintOSC1MB();
    void caption( uint r, uint c, const GFXfont* f, String s1, String s2, uint pp );
};

// - - - - - - - - - - - - -
// Help
class helpPage : public Page
{
  public:
    void init();
    void spinHandler(int8_t address, int8_t spin);
    void pressHandler(int8_t address, VelociBus::BUTTON_EVENT event);     
    KNOBINFO_T knobInfo(uint knob_index);
  public:
    uint parent;
    uint button;
};

// - - - - - - - - - - - - -
// Screen saver
class saverPage : public Page
{
  public:
    void init();
    void loop();
    KNOBINFO_T knobInfo(uint knob_index);
  public:
    uint parent;
    uint32_t tm;
};

// - - - - - - - - - - - - -
// Splash
class splashPage : public Page
{
  public:
    void init();
    void loop();
    void spinHandler(int8_t address, int8_t spin);
    void pressHandler(int8_t address, VelociBus::BUTTON_EVENT event);     
    KNOBINFO_T knobInfo(uint knob_index);
  private:
    uint32_t tm;
};

//------------------------------------------------------

// VelociBus breadboard knob addressing
// (1) (4)
// (2) (5)
// (3) (6)
//   (0)
#define KNOB_NAV  0 // receive address
#define KNOB_UL   1
#define KNOB_ML   2
#define KNOB_LL   3
#define KNOB_UR   4
#define KNOB_MR   5
#define KNOB_LR   6
#define KNOB_TX(rx_addr) (6-rx_addr) // transmit address is reverse of receive address

// Page index
//   dial selectable pages
#define PG_HOME       0 
#define PG_OSC1       1 
#define PG_OSC2       2
#define PG_BJ_MIX     3
#define PG_BJ_FILTER  4
#define PG_BLIPPO     5
#define PG_JAM        6
#define SCROLLABLE_PAGES 7
//   settings pages
#define PG_ABOUT      7
//   sub pages
#define PG_HELP       8
#define PG_SPLASH     9
#define PG_SAVER     10
#define NUM_PAGES    11

class PageController
{
  public:
    void fatalError( uint blinks, String msg );

    void help( uint pg, int8_t button_index )
    {
      help_page.parent = pg;
      help_page.button = button_index;
      start( PG_HELP );
    }

    void setBrightness( byte b )
    {
      analogWrite( BL_PIN, b );
    }  

    void start(uint page_index) // show a page
    {
      if (page_index >= NUM_PAGES) return; // failsafe
      // 7-seg single digit on navigation knob
      switch (page_index)
      { case PG_ABOUT: vbus.rotor9LED(KNOB_TX(KNOB_NAV), 10);       break; // A
        case PG_JAM  : vbus.rotor9LEDspecial(KNOB_TX(KNOB_NAV), 3); break; // J
        case PG_SPLASH: 
        case PG_HELP : vbus.rotor9LEDspecial(KNOB_TX(KNOB_NAV), 2); break; // H
        case PG_SAVER: break; 
        default      : vbus.rotor9LED(KNOB_TX(KNOB_NAV), getPageSeqIndex(page_index)+1 );
      }
      // light up active knobs
      vbus.rotorLED( KNOB_TX(KNOB_UL), ui_page[page_index]->knobInfo(0).color );
      vbus.rotorLED( KNOB_TX(KNOB_ML), ui_page[page_index]->knobInfo(1).color );
      vbus.rotorLED( KNOB_TX(KNOB_LL), ui_page[page_index]->knobInfo(2).color );
      vbus.rotorLED( KNOB_TX(KNOB_UR), ui_page[page_index]->knobInfo(3).color );
      vbus.rotorLED( KNOB_TX(KNOB_MR), ui_page[page_index]->knobInfo(4).color );
      vbus.rotorLED( KNOB_TX(KNOB_LR), ui_page[page_index]->knobInfo(5).color );
      // set knob acceleration
      vbus.rotorConfig( KNOB_TX(KNOB_UL), ui_page[page_index]->knobInfo(0).acceleration, 0 );
      vbus.rotorConfig( KNOB_TX(KNOB_ML), ui_page[page_index]->knobInfo(1).acceleration, 0 );
      vbus.rotorConfig( KNOB_TX(KNOB_LL), ui_page[page_index]->knobInfo(2).acceleration, 0 );
      vbus.rotorConfig( KNOB_TX(KNOB_UR), ui_page[page_index]->knobInfo(3).acceleration, 0 );
      vbus.rotorConfig( KNOB_TX(KNOB_MR), ui_page[page_index]->knobInfo(4).acceleration, 0 );
      vbus.rotorConfig( KNOB_TX(KNOB_LR), ui_page[page_index]->knobInfo(5).acceleration, 0 );
      // show the screen      
      ui_page[current_page = page_index]->init();
    }
    
    void run() // poll for input
    {
      random(); // for screen saver
      
      // watch velocibus for knob inputs
      vbus_packet pkt;
      if (vbus.poll( &pkt ))
      {
        saver_keepalive = true;
        if (current_page == PG_SAVER) // wake up
          start( current_page = saver_page.parent );
        else
        { // decode packet
          if (pkt.address == KNOB_NAV)
            // handle global page select knob
            switch (pkt.data)
            { case 0x20: // Press
                // toggles About page
                if (current_page == PG_HOME)
                  start(current_page = PG_ABOUT );
                else if (current_page == PG_ABOUT)
                  start(current_page = PG_HOME );
                else if (current_page < SCROLLABLE_PAGES)
                  start(current_page = PG_HOME);
                else // let the page handle it
                  ui_page[current_page]->pressHandler(pkt.address,VelociBus::press);
                break;  
  
              case 0x21: // Long press
                // toggle MIDI value display
                show_midi_values = !show_midi_values;
                start(current_page);
                for (uint i=0; i < PATCH_SIZE; i++) DB.live_data.patch[i] = 0x80; // refresh synth
                break;              
                  
              case 0x1F: break; // release
              
              default  : // spin
              { int8_t spin = pkt.data; // 6 bit signed int
                if (spin & 0x20) spin |= 0xC0; // extend sign to 8 bits
                if (current_page < SCROLLABLE_PAGES)
                  start(current_page = bumpPage(current_page, (spin > 0 ? 1 : -1)) );
                else // send to page
                   ui_page[current_page]->spinHandler(pkt.address,spin);
              }  
            }
          else // send to current page
            switch (pkt.data)
            { case 0x20: ui_page[current_page]->pressHandler(pkt.address,VelociBus::press);   break;
              case 0x21: ui_page[current_page]->pressHandler(pkt.address,VelociBus::hold);    break;
              case 0x1F: ui_page[current_page]->pressHandler(pkt.address,VelociBus::release); break;
              default  : 
              { int8_t spin = pkt.data; // 6 bit signed int
                if (spin & 0x20) spin |= 0xC0; // extend sign to 8 bits
                ui_page[current_page]->spinHandler(pkt.address,spin);
              }
            }
        }
      }
              
      // watch for patch data changes & update screen
      ui_page[current_page]->loop();

      // screen saver
      //   keepalive
      if (saver_keepalive)
      { saver_keepalive = false;
        saver_tm = millis();
      }      
      //   activation
      if (current_page != PG_SAVER)
        if (millis() - saver_tm >= SCREENSAVE_MSEC)
        { saver_page.parent = current_page;
          start(current_page = PG_SAVER);      
        }
    }

  private:
    const byte benjolin_page_sequence[6] = { PG_HOME, PG_OSC1, PG_OSC2, PG_BJ_MIX, PG_BJ_FILTER, PG_JAM };
    const byte blippo_page_sequence  [5] = { PG_HOME, PG_OSC1, PG_OSC2, PG_BLIPPO,               PG_JAM };

    int bumpPage( uint page_num, int incr ) // bump page in sequence for current mode
    {
      int    num_pages = (DB.ui_data.patch[ppMODE] == MODE_BENJOLIN) ? 6 : 5;
      const byte *pSeq = (DB.ui_data.patch[ppMODE] == MODE_BENJOLIN) ? benjolin_page_sequence : blippo_page_sequence;
      int       seq_ix = (getPageSeqIndex(page_num) + incr + num_pages) % num_pages;
      return pSeq[seq_ix];
    }

    int getPageSeqIndex( uint page_num )
    {
      int    num_pages = (DB.ui_data.patch[ppMODE] == MODE_BENJOLIN) ? 6 : 5;
      const byte *pSeq = (DB.ui_data.patch[ppMODE] == MODE_BENJOLIN) ? benjolin_page_sequence : blippo_page_sequence;
      // figure out what the current sequence index is
      for (int seq_ix=0; seq_ix < num_pages; seq_ix++) if (pSeq[seq_ix] == page_num) return seq_ix;
      return -1; // error
    }             

  public:
    // create the pages
    //   scrollable pages
    HomePage       home_page;
    OSC1Page       osc1_page;
    OSC2Page       osc2_page;
    BJ_MixPage     bj_mix_page;
    BJ_FilterPage  bj_filter_page;
    BlippoPage     blippo_page;
    JamPage        jam_page;
    //   special pages
    aboutPage      about_page;
    helpPage       help_page;
    splashPage     splash_page;
    saverPage      saver_page;
    // make a table of pages
    Page *ui_page[NUM_PAGES] = 
    { &home_page, &osc1_page, &osc2_page, &bj_mix_page, &bj_filter_page, &blippo_page, &jam_page,
      &about_page, &help_page, &splash_page, &saver_page
    };
    uint current_page = PG_SPLASH;
    // screen saver
    boolean  saver_keepalive = true;
    uint32_t saver_tm;
    // debug
    boolean show_midi_values = false;
};

#endif
