//-------------------------------------------------------------------------
// Teensy4 Drone
// 2023.05.15 RSP
//
// Target: Teensy4.x
//         Serial(usb) = diagnostic serial monitor
//         Serial1 = not used
//         Serial2 = audio shield
//         Serial3 = MIDI
//         SD = BUILTIN_SDCARD  NOT USED
//         Audio Shield - REMOVE 0.1uF CAP ON PIN 15 (Rx3)
//
// Connects to controller via serial midi.
//
// Status LED: blinks if low audio memory
//-------------------------------------------------------------------------

//#define DEBUG_CONSOLE // for memory diagnostics on serial monitor

// Teensy4 memory diagnostics
extern unsigned long _heap_start;
extern unsigned long _heap_end;
extern char *__brkval;
int freeram() { return (char *)&_heap_end - __brkval; }

#include <MIDI.h>
#include "common.h"
#include "fscale.h"

#define STATUS_LED_PIN 2 // flashes error code

//-------------------------------------------------------------------------
// MIDI serial connection to controller

MIDI_CREATE_INSTANCE(HardwareSerial, Serial3, MIDI);

//-------------------------------------------------------------------------
// GUItool code

#include "effect_rungler.h"
#include "effect_comparator.h"
#include "effect_samplehold.h"
#include "effect_ensemble.h"

#include <Audio.h>
#include <Wire.h>
#include <SPI.h>
#include <SD.h>
#include <SerialFlash.h>

//-------------------------------------------------------------------------

// GUItool: begin automatically generated code
AudioEffectRungler        rungler;        //xy=466,543
AudioEffectRungler        rungler2; //xy=572,744
AudioEffectSampleAndHold  sampleHold; //xy=507,183
AudioEffectComparator     compare;        //xy=593,408
AudioEffectEnsemble       ensemble;     //xy=1431,137
AudioMixer4              mixer2;         //xy=105,668
AudioMixer4              mixer1;         //xy=112,382
AudioMixer4              mixer_SH_DATA; //xy=167,40
AudioMixer4              mixer_SH_TRIG; //xy=242,107
AudioSynthWaveformModulated waveformMod2;   //xy=257,675
AudioSynthWaveformModulated waveformMod1;   //xy=264,389
AudioMixer4              mixer_SH; //xy=588,219
AudioSynthWaveformDc     dc_FMOD;            //xy=608,584
AudioMixer4              mixer_FREQ; //xy=763,434
AudioMixer4              mixer_FMOD;         //xy=765,503
AudioFilterLadder        ladder1; //xy=769,220
AudioFilterLadder        ladder2; //xy=770,273
AudioAmplifier           amp_NEG;           //xy=912,222
AudioFilterStateVariable filter;        //xy=914,464
AudioMixer4              mixer_OUT;         //xy=1074,394
AudioOutputI2S           i2s1;           //xy=1101,490
AudioConnection          patchCord1(mixer2, 0, waveformMod2, 0);
AudioConnection          patchCord2(mixer1, 0, waveformMod1, 0);
AudioConnection          patchCord3(mixer_SH_DATA, 0, sampleHold, 0);
AudioConnection          patchCord4(mixer_SH_TRIG, 0, sampleHold, 1);
AudioConnection          patchCord5(waveformMod2, 0, rungler, 1);
AudioConnection          patchCord6(waveformMod2, 0, compare, 1);
AudioConnection          patchCord7(waveformMod2, 0, mixer1, 2);
AudioConnection          patchCord8(waveformMod2, 0, rungler2, 0);
AudioConnection          patchCord9(waveformMod2, 0, mixer_SH_DATA, 0);
AudioConnection          patchCord10(waveformMod2, 0, mixer_FMOD, 1);
AudioConnection          patchCord11(waveformMod2, 0, mixer_FREQ, 2);
AudioConnection          patchCord12(waveformMod1, 0, compare, 0);
AudioConnection          patchCord13(waveformMod1, 0, rungler, 0);
AudioConnection          patchCord14(waveformMod1, 0, rungler2, 1);
AudioConnection          patchCord15(waveformMod1, 0, mixer_SH_TRIG, 0);
AudioConnection          patchCord16(waveformMod1, 0, mixer2, 1);
AudioConnection          patchCord17(waveformMod1, 0, mixer_FREQ, 1);
AudioConnection          patchCord18(rungler, 0, mixer1, 1);
AudioConnection          patchCord19(rungler, 0, mixer_SH, 1);
AudioConnection          patchCord20(rungler, 0, mixer_SH_DATA, 1);
AudioConnection          patchCord21(rungler, 0, mixer2, 0);
AudioConnection          patchCord22(rungler, 0, mixer_FMOD, 0);
AudioConnection          patchCord23(rungler2, 0, mixer_SH, 2);
AudioConnection          patchCord24(rungler2, 0, mixer_SH_DATA, 2);
AudioConnection          patchCord25(rungler2, 0, mixer2, 2);
AudioConnection          patchCord26(sampleHold, 0, mixer_SH, 0);
AudioConnection          patchCord27(sampleHold, 0, mixer1, 0);
AudioConnection          patchCord28(compare, 0, ladder2, 0);
AudioConnection          patchCord29(compare, 0, ladder1, 0);
AudioConnection          patchCord30(compare, 0, mixer_SH_TRIG, 1);
AudioConnection          patchCord31(compare, 0, mixer_FREQ, 0);
AudioConnection          patchCord32(mixer_SH, 0, ladder1, 1);
AudioConnection          patchCord33(mixer_SH, 0, ladder2, 1);
AudioConnection          patchCord34(dc_FMOD, 0, mixer_FMOD, 2);
AudioConnection          patchCord35(mixer_FREQ, 0, filter, 0);
AudioConnection          patchCord36(mixer_FMOD, 0, filter, 1);
AudioConnection          patchCord37(ladder1, amp_NEG);
AudioConnection          patchCord38(ladder2, 0, mixer_OUT, 1);
AudioConnection          patchCord39(amp_NEG, 0, mixer_OUT, 0);
AudioConnection          patchCord40(filter, 1, mixer_OUT, 2);
AudioConnection          patchCord41(dc_FMOD, 0, mixer1, 3);
AudioConnection          patchCord42(dc_FMOD, 0, mixer2, 3);

// dynamic connections

//AudioConnection          patchCord43(mixer_OUT, 0, i2s1, 1);
//AudioConnection          patchCord44(mixer_OUT, 0, i2s1, 0);

//AudioConnection          patchCord45(mixer_OUT, 0, ensemble, 0);
//AudioConnection          patchCord46(ensemble, 0, i2s1, 0);
//AudioConnection          patchCord47(ensemble, 1, i2s1, 1);

AudioControlSGTL5000     sgtl5000_1;     //xy=588,32
// GUItool: end automatically generated code

// dynamic audio connections
#include <vector>
std::vector<AudioConnection*> connections; 

void setConnections(boolean stereo)
{
  for (auto x : connections) delete x;
  connections.clear();
  if (stereo)
  { // insert ensemble object
    connections.push_back(new AudioConnection(mixer_OUT,0, ensemble,0));
    connections.push_back(new AudioConnection(ensemble, 0, i2s1, 0));
    connections.push_back(new AudioConnection(ensemble, 1, i2s1, 1));
  }
  else
  { // straight out
    connections.push_back(new AudioConnection(mixer_OUT,0,  i2s1, 0));
    connections.push_back(new AudioConnection(mixer_OUT, 1, i2s1, 1));
  } 
}

//-------------------------------------------------------------------------

uint  gMode   = MODE_BENJOLIN;
float gVolume = 0;

//-------------------------------------------------------------------------
// helpers

void memoryDiags(boolean force = false)
{ 
  // diagnostic report every few seconds
  { static uint32_t tm = millis();
    if (force || millis() - tm > 60000UL) 
    { tm = millis();
      Serial.print(F("HEAP:"));
      Serial.print(freeram());
      Serial.print(F(" CPU:"));
      Serial.print(AudioProcessorUsage());
      Serial.print(F("/"));
      Serial.print(AudioProcessorUsageMax());
      Serial.print(F(" MEM:"));
      Serial.print(AudioMemoryUsage());    
      Serial.print(F("/"));
      Serial.println(AudioMemoryUsageMax());
    }
  }
}

void setVolume()
{
  switch (gMode)
  {
    case MODE_BLIPPO1:
    case MODE_BLIPPO2: 
    case MODE_BLIPPO3:
      mixer_OUT.gain(0,gVolume);
      mixer_OUT.gain(1,gVolume);
      mixer_OUT.gain(2,0);
      break;

    default: // MODE_BENJOLIN
      mixer_OUT.gain(0,0);
      mixer_OUT.gain(1,0);
      mixer_OUT.gain(2,gVolume);
  }
}

void setMode() // set Blippo mode
{
  switch (gMode)
  {
    case MODE_BLIPPO1:
      mixer_SH_TRIG.gain(0,1); // osc1 -> SH_TRIG
      mixer_SH_DATA.gain(0,1); // osc2 -> SH_DATA

      mixer_SH_DATA.gain(1,0);
      mixer_SH_DATA.gain(2,0);
      mixer_SH_TRIG.gain(1,0);
      break;

    case MODE_BLIPPO2: 
      mixer_SH_TRIG.gain(1,1); // compare -> SH_TRIG
      mixer_SH_DATA.gain(0,1); // osc2 -> SH_DATA

      mixer_SH_DATA.gain(1,0);
      mixer_SH_DATA.gain(2,0);
      mixer_SH_TRIG.gain(0,0);
      break;

    case MODE_BLIPPO3:
      mixer_SH_TRIG.gain(0,1);   // osc1 -> SH_TRIG
      mixer_SH_DATA.gain(1,0.5); // runglers -> SH_DATA
      mixer_SH_DATA.gain(2,0.5);
      
      mixer_SH_TRIG.gain(1,0);
      mixer_SH_DATA.gain(0,0);
      break;

    default: // MODE_BENJOLIN
      break;
  }
  setVolume();
}
    
//-------------------------------------------------------------------------
// Program start

#define INITMIXER(m) {m.gain(0,0);m.gain(1,0);m.gain(2,0);m.gain(3,0);}

FLASHMEM void setup() 
{
#ifdef DEBUG_CONSOLE  
  // serial monitor diagnostics
  Serial.begin(115200); // USB serial monitor
  delay(5000);
  Serial.println("Teensy Drone"); 
  memoryDiags(true);
#endif

  // configure audio subsystem
  AudioMemory(100);
  sgtl5000_1.enable();
  sgtl5000_1.volume(1.0);

  // default is 10 octaves of modulation control
  waveformMod1.begin(WAVEFORM_TRIANGLE);
  waveformMod1.amplitude(1);  
  waveformMod2.begin(WAVEFORM_TRIANGLE);
  waveformMod2.amplitude(1);

  ladder1.octaveControl(2.5);
  ladder2.octaveControl(2.5);
  ladder1.frequency(1000);
  ladder2.frequency(1000);

  INITMIXER(mixer1);
  INITMIXER(mixer2);
  INITMIXER(mixer_FREQ);
  INITMIXER(mixer_FMOD);
  INITMIXER(mixer_SH_DATA);
  INITMIXER(mixer_SH_TRIG);
  INITMIXER(mixer_SH);
  INITMIXER(mixer_OUT);

  dc_FMOD.amplitude(1.0);

  amp_NEG.gain(-1);

  filter.resonance(0.707);
  filter.frequency(1000);

  setMode();
  setConnections(false); // default to mono output
  
  // MIDI serial
  MIDI.begin(MIDI_CHANNEL_OMNI); // listen on all channels
  MIDI.turnThruOff();
  MIDI.setHandleControlChange(myControlChange);
  MIDI.setHandleProgramChange(myProgramChange);
  MIDI.setHandleNoteOff      (myNoteOff);
  MIDI.setHandleNoteOn       (myNoteOn);
    
  // status LED
  pinMode( STATUS_LED_PIN, OUTPUT );  digitalWrite( STATUS_LED_PIN, HIGH );
}
//-------------------------------------------------------------------------
// Bigloop

void loop(void) 
{
  // trigger MIDI callbacks
  if (MIDI.read()) // triggers midi callbacks
  { 
#ifdef DEBUG_CONSOLE  
  if (MIDI.getType() != 0xFE) // filter clock messages
  { Serial.print("MIDI: ");
    Serial.print(MIDI.getType(),HEX); Serial.print(" ");
    Serial.print(MIDI.getChannel()); Serial.print(" ");
    Serial.print(MIDI.getData1()); Serial.print(" ");
    Serial.println(MIDI.getData2());
  }
#endif    
  }

#ifdef DEBUG_CONSOLE  
  // diagnostic console
  { memoryDiags(); // periodic report
    int c = Serial.read();
    switch (c)
    { case -1: break;
      case 'D': case 'd': // dump current state
        break;
      case 'M': case 'm': // memory diags
        memoryDiags(true);
        break;
      case '0': gMode = 0; setMode(); break;
      case '1': gMode = 1; setMode(); break;
      case '2': gMode = 2; setMode(); break;
      case '3': gMode = 3; setMode(); break;
    }
  }
#endif 

  // watch for low memory
  if (AudioMemoryUsageMax() > 90)
    for (;;) // halt and blink
    { digitalWrite( STATUS_LED_PIN, !digitalRead(STATUS_LED_PIN) );
      delay(250);
    }
}

//-------------------------------------------------------------------------
// MIDI callbacks

#define FMOD_GAIN(n) (n/256.0)

// - - - - - - - - - - - - -
// MIDI note on/off
void myNoteOn([[maybe_unused]]byte inChannel, byte inNoteNumber, byte inVelocity) 
{
#ifdef DEBUG_CONSOLE  
Serial.print("NoteOn ");Serial.print(inChannel);Serial.print("=");Serial.println(inNoteNumber);
#endif
  // split keyboard
  if (inNoteNumber > 63)
       { mixer1.gain(3,FMOD_GAIN(inNoteNumber-64)); mixer2.gain(3,0); }
  else { mixer2.gain(3,FMOD_GAIN(64-inNoteNumber)); mixer1.gain(3,0); }
}
void myNoteOff([[maybe_unused]]byte inChannel, [[maybe_unused]]byte inNoteNumber, [[maybe_unused]]byte inVelocity)
{ 
  mixer1.gain(3,0);
  mixer2.gain(3,0);
}

// - - - - - - - - - - - - -
// MIDI program change (select FX type)
void myProgramChange(byte channel, byte program) // channel 1..16, program 0..127 (constrained to num modes)
{
  if (program >= NUM_MODES) return; // failsafe
  gMode = program;
  setMode();
}

// - - - - - - - - - - - - -
// MIDI control change (set FX property)
void myControlChange(byte channel, byte control, byte value) // channel = 1..16
{
#ifdef DEBUG_CONSOLE  
Serial.print("CC ");Serial.print(control);Serial.print("=");Serial.println(value);
#endif

  switch (control)
  {

// common controls
      
    case ccVOLUME:
      gVolume = SCALE(value);
      setVolume();
      break;

    case ccSTEREO:
      setConnections( value > 0 );
      break;
      
    case ccOSC1_FREQ:
    { float f = OSC_FREQ(value);
#ifdef DEBUG_CONSOLE  
Serial.print(f); Serial.println("Hz");    
#endif
      waveformMod1.frequency( f );
      break;
    }
    
    case ccOSC2_FREQ:
    { float f = OSC_FREQ(value);
#ifdef DEBUG_CONSOLE  
Serial.print(f); Serial.println("Hz");    
#endif
      waveformMod2.frequency( f );
      break;
    }
    
    case ccOSC1_SH_MOD:
      mixer1.gain( 0, SCALE(value) );
      break;
    
    case ccOSC1_RUNGLER_MOD:
      mixer1.gain( 1, SCALE(value) );
      break;
    
    case ccOSC1_CROSS_MOD:
      mixer1.gain( 2, SCALE(value) );
      break;
    
    case ccOSC2_RUNGLER1_MOD:
      mixer2.gain( 0, SCALE(value) );
      break;
    
    case ccOSC2_RUNGLER2_MOD:
      mixer2.gain( 1, SCALE(value) );
      break;
    
    case ccOSC2_CROSS_MOD:
      mixer2.gain( 2, SCALE(value) );
      break;

// benjolin
      
    case ccCOMPARE_MIX:   // compare -> filter
      mixer_FREQ.gain( 0, SCALE(value) );
      break;
    
    case ccOSC1_MIX:      // OSC1 -> filter
      mixer_FREQ.gain( 1, SCALE(value) );
      break;
    
    case ccOSC2_MIX:      // OSC2 -> filter
      mixer_FREQ.gain( 2, SCALE(value) );
      break;

    case ccFILTER_FREQ:   // filter frequency
      filter.frequency( FILTER_FREQ(value) );
      break;
    
    case ccFILTER_Q:      // filter resonance
      filter.resonance( FILTER_RESONANCE(value) );
      break;
    
    case ccFMOD_RUNGLER1: // rungler -> filter modulation
      mixer_FMOD.gain( 0, SCALE(value) );
      break;
    
    case ccFMOD_OSC2:     // OSC2 -> filter modulation
      mixer_FMOD.gain( 1, SCALE(value) );
      break;
    
    case ccFMOD_DC:       // knob -> filter modulation
      mixer_FMOD.gain( 2, SCALE(value) );
      break;

// blippo

    case ccSH_MIX:        // compare mix
      mixer_SH.gain( 0, SCALE(value) );
      break;
      
    case ccSH_RUNGLER1:   // rungler1 mix
      mixer_SH.gain( 1, SCALE(value) );
      break;
      
    case ccSH_RUNGLER2:   // rungler2 mix
      mixer_SH.gain( 2, SCALE(value) );
      break;
    
    case ccLADDER1_FREQ:  // ladder1 frequency
      ladder1.frequency( LADDER_FREQ(value) );
      break;
    
    case ccLADDER2_FREQ:  // ladder2 frequency
      ladder2.frequency( LADDER_FREQ(value) );
      break;

    case ccLADDER_Q:      // ladder resonance (both)
      ladder1.resonance( LADDER_RESONANCE(value) );
      ladder2.resonance( LADDER_RESONANCE(value) );
  }
}
