/* Audio Library for Teensy 3.X
   Portions Copyright (c) 2019, Alexander Davis info@matrixwide.com
   Portions Copyright (c) 2021, Vince R. Pearson
    
   Permission is hereby granted, free of charge, to any person obtaining a copy
   of this software and associated documentation files (the "Software"), to deal
   in the Software without restriction, including without limitation the rights
   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
   copies of the Software, and to permit persons to whom the Software is
   furnished to do so, subject to the following conditions:

   The above copyright notice, development funding notice, and this permission
   notice shall be included in all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
   THE SOFTWARE.
*/

// My take on a string machine ensemble chorus sound like the Roland RS-202
// or Lowrey Symphonic Strings.
// It takes the approach of 6.0 Hz and 0.6 Hz sinewaves making up an LFO,
// which then modulates three delay lines 120 degrees apart in the LFO waveform.


// V. Pearson: Implemented delay buffer interpolation with floating point fractional indices and using a larger table
// to reduce audible delay sweep quantization noise.
// The fractional indices are stored in flash memory instead of being calculated at runtime and stored in RAM.
// #define LARGE_ENSEMBLE_LFO_TABLE uses a larger lfo table with only delay buffer interpolation.
// Without it lfo table and delay buffer interpolation are implemented.

#include <Arduino.h>
#include "effect_ensemble.h"
#include "utility/dspinst.h"
#include "arm_math.h"

AudioEffectEnsemble::AudioEffectEnsemble() : AudioStream(1, inputQueueArray){
  memset(delayBuffer, 0, sizeof(delayBuffer));

  // input index
  inIndex = 0;
  // output indexes
  // default to center of buffer
  outIndex1 = 512;
  outIndex2 = 512;
  outIndex3 = 512;
  outIndex4 = 512;
  outIndex5 = 512;
  outIndex6 = 512;
  // lfo index
  // seprated by sixths to approximate 60 degree phase relationship
  lfoIndex1 = 0;
  lfoIndex2 = 245;
  lfoIndex3 = 490;
  lfoIndex4 = 735;
  lfoIndex5 = 980;
  lfoIndex6 = 1225;
  // lfo rate counter
  lfoCount = 0;
  // input index offset
  offset1 = 0;
  offset2 = 0;
  offset3 = 0;
  offset4 = 0;
  offset5 = 0;
  offset6 = 0;
}

// TODO: move this to one of the data files, use in output_adat.cpp, output_tdm.cpp, etc
static const audio_block_t zeroblock = {
  0, 0, 0, {
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
#if AUDIO_BLOCK_SAMPLES > 16
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
#endif
#if AUDIO_BLOCK_SAMPLES > 32
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
#endif
#if AUDIO_BLOCK_SAMPLES > 48
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
#endif
#if AUDIO_BLOCK_SAMPLES > 64
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
#endif
#if AUDIO_BLOCK_SAMPLES > 80
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
#endif
#if AUDIO_BLOCK_SAMPLES > 96
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
#endif
#if AUDIO_BLOCK_SAMPLES > 112
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
#endif
  }
};

//Rate is in Hz
void AudioEffectEnsemble::lfoRate(float rate)
{
  //Assumes COUNTS_PER_LFO is giving 6Hz
  countsPerLfo = round((COUNTS_PER_LFO * 6) / rate);
  if (countsPerLfo < 1) {
    countsPerLfo = 1;
  }
}

void AudioEffectEnsemble::update(void)
{
  const audio_block_t *block;
  audio_block_t *outblock;
  audio_block_t *outblockB;
  uint16_t i;
  int32_t sum;

  outblock = allocate();
  outblockB = allocate();
  if ((!outblock) || (!outblockB)) {
    audio_block_t *tmp = receiveReadOnly(0);
    if (tmp) release(tmp);
    return;
  }
  block = receiveReadOnly(0);
  if (!block)
    block = &zeroblock;

  // buffer the incoming block
  for (i = 0; i < AUDIO_BLOCK_SAMPLES; i++){
    // wrap the input index
    inIndex++;
    if (inIndex > (ENSEMBLE_BUFFER_SIZE - 1))
      inIndex = 0;
    delayBuffer[inIndex] = block->data[i];
  }
  // re-load the block with the delayed data
  for (i = 0; i < AUDIO_BLOCK_SAMPLES; i++) {
    // advance the wavetable indexes every counts per LFO
    // so the LFO modulates at the correct rate
    lfoCount++;
    if (lfoCount > countsPerLfo){
      // wrap the lfo index
      lfoIndex1++;
      if (lfoIndex1 > (LFO_SIZE - 1))
        lfoIndex1 = 0;
        
      lfoIndex2++;
      if (lfoIndex2 > (LFO_SIZE - 1))
        lfoIndex2 = 0;
        
      lfoIndex3++;
      if (lfoIndex3 > (LFO_SIZE - 1))
        lfoIndex3 = 0;
      
      lfoIndex4++;
      if (lfoIndex4 > (LFO_SIZE - 1))
        lfoIndex4 = 0;
        
      lfoIndex5++;
      if (lfoIndex5 > (LFO_SIZE - 1))
        lfoIndex5 = 0;        

      lfoIndex6++;
      if (lfoIndex6 > (LFO_SIZE - 1))
        lfoIndex6 = 0;        

      // reset the counter
      lfoCount = 0;
    }

    // wrap the output index
    outIndex1++;
    if (outIndex1 > (ENSEMBLE_BUFFER_SIZE - 1))
      outIndex1 = 0;

    outIndex2++;
    if (outIndex2 > (ENSEMBLE_BUFFER_SIZE - 1))
      outIndex2 = 0;

    outIndex3++;
    if (outIndex3 > (ENSEMBLE_BUFFER_SIZE - 1))
      outIndex3 = 0;

    outIndex4++;
    if (outIndex4 > (ENSEMBLE_BUFFER_SIZE - 1))
      outIndex4 = 0;

    outIndex5++;
    if (outIndex5 > (ENSEMBLE_BUFFER_SIZE - 1))
      outIndex5 = 0;

    outIndex6++;
    if (outIndex6 > (ENSEMBLE_BUFFER_SIZE - 1))
      outIndex6 = 0;

     
#ifdef LARGE_ENSEMBLE_LFO_TABLE
    offset1 = lfoTable[lfoIndex1];
    offset2 = lfoTable[lfoIndex2];
    offset3 = lfoTable[lfoIndex3];
    offset4 = lfoTable[lfoIndex4];
    offset5 = lfoTable[lfoIndex5];
    offset6 = lfoTable[lfoIndex6];
#else
    offset1 = lfoLookup(lfoIndex1);
    offset2 = lfoLookup(lfoIndex2);
    offset3 = lfoLookup(lfoIndex3);
    offset4 = lfoLookup(lfoIndex4);
    offset5 = lfoLookup(lfoIndex5);
    offset6 = lfoLookup(lfoIndex6);	
#endif
    

    // add the delayed samples and scale

    sum=(int32_t)interpBuffer((float)outIndex1 + offset1);
    sum+=(int32_t)interpBuffer((float)outIndex2 + offset2);
    sum+=(int32_t)interpBuffer((float)outIndex3 + offset3);
    sum+=(int32_t)interpBuffer((float)outIndex4 + offset4);
    sum+=(int32_t)interpBuffer((float)outIndex5 + offset5);
    sum+=(int32_t)interpBuffer((float)outIndex6 + offset6);
    sum=sum/6;
    outblock->data[i]=(uint16_t)sum;

    sum=(int32_t)interpBuffer((float)outIndex1 + offset1+PHASE_90);
    sum+=(int32_t)interpBuffer((float)outIndex2 + offset2+PHASE_90);
    sum+=(int32_t)interpBuffer((float)outIndex3 + offset3+PHASE_90);
    sum+=(int32_t)interpBuffer((float)outIndex4 + offset4+PHASE_90);
    sum+=(int32_t)interpBuffer((float)outIndex5 + offset5+PHASE_90);
    sum+=(int32_t)interpBuffer((float)outIndex6 + offset6+PHASE_90);
    sum=sum/6;
    outblockB->data[i]=(uint16_t)sum;
  }

  transmit(outblock, 0);
  transmit(outblockB, 1);
  release(outblock);
  release(outblockB);
  if (block != &zeroblock) release((audio_block_t *)block);


  return;

}

// returns interpolated value from delay buffer between int(findex) and int(findex+1) using the fractional part of findex.
int16_t AudioEffectEnsemble::interpBuffer(float findex)
{
  float fintIndex;
  float frac;
  float y0,y1;
  int16_t index1,index2;
  
  // get integer and fractional part of index.
  frac=modff(findex,&fintIndex);
  index1=(int16_t)fintIndex;
  
  // wrap integer index if beyond start or end.
  if(index1>(ENSEMBLE_BUFFER_SIZE-1)) index1-=ENSEMBLE_BUFFER_SIZE;
  else if(index1<0) index1+=ENSEMBLE_BUFFER_SIZE;
  
  // check if next sample index needs to be wrapped.
  index2=index1+1;
  if(index2>(ENSEMBLE_BUFFER_SIZE-1)) index2-=ENSEMBLE_BUFFER_SIZE;
  else if(index2<0) index2+=ENSEMBLE_BUFFER_SIZE;
  
  // calculate interpolated value from delay buffer.
  y0=(float)delayBuffer[index1];
  y1=(float)delayBuffer[index2];
  return (int16_t)round(y0+frac*(y1-y0));
  
}
#ifndef LARGE_ENSEMBLE_LFO_TABLE
float AudioEffectEnsemble::lfoLookup(int16_t lfoIndex)
{
  // Linear interpolating table lookup. Assumes a table 4 times smaller than the full index range.
  // The lower two bits are used for interpolating between table values y0 and y1.
  // The upper bits are a direct index (y1) into the lfo table. y0 is the last entry lookup.
  // This effectively makes the table 4 times larger with some interpolation error.
  // The function assumes that the index increments by only 1 each on subsequent call.
  float frac,y0,y1;
  int16_t index0,index1;
  
  index0=lfoIndex>>2; // get integer index
  index1=index0+1;
  if(index1>LFO_SAMPLES-1) index1=0;
  frac= (float)(lfoIndex&0x3)*0.25; // calculate interpolating fraction between samples.
  y0=lfoTable[index0];
  y1=lfoTable[index1];
  return y0+(y1-y0)*frac;
}
#endif



#ifdef LARGE_ENSEMBLE_LFO_TABLE
// Table created from Octave script using original table function:
// y=((sin(((2.0 * pi)/LFO_SAMPLES) * x) * LFO_RANGE) / 2.0) + (((sin(((20.0 * pi)/LFO_SAMPLES) * x)) * LFO_RANGE) / 3.0);
// COUNTS_PER_LFO is divided by the same same factor to maintain the same table sweep rate.
// This ~24KB table could be reduced by a factor of two due to its symmetry with some extra code overhead but since only 3% of the
// program space is being used at this time there is no significant benefit in implementing it.

const float PROGMEM AudioEffectEnsemble::lfoTable[LFO_SAMPLES]={
        0.000000,0.450572,0.901100,1.351538,1.801842,2.251966,2.701867,3.151499,
        3.600817,4.049778,4.498336,4.946446,5.394065,5.841148,6.287650,6.733528,
        7.178736,7.623231,8.066968,8.509903,8.951994,9.393194,9.833462,10.272753,
        10.711024,11.148232,11.584332,12.019283,12.453040,12.885562,13.316805,13.746726,
        14.175284,14.602436,15.028140,15.452353,15.875035,16.296143,16.715636,17.133473,
        17.549612,17.964014,18.376636,18.787439,19.196383,19.603426,20.008531,20.411656,
        20.812762,21.211811,21.608763,22.003580,22.396223,22.786654,23.174835,23.560729,
        23.944297,24.325504,24.704311,25.080682,25.454582,25.825973,26.194820,26.561088,
        26.924741,27.285745,27.644065,27.999666,28.352515,28.702579,29.049822,29.394214,
        29.735720,30.074309,30.409948,30.742606,31.072252,31.398854,31.722381,32.042805,
        32.360093,32.674218,32.985149,33.292858,33.597316,33.898496,34.196368,34.490907,
        34.782084,35.069873,35.354249,35.635185,35.912655,36.186635,36.457100,36.724026,
        36.987388,37.247164,37.503330,37.755864,38.004743,38.249946,38.491451,38.729237,
        38.963284,39.193571,39.420080,39.642790,39.861682,40.076739,40.287943,40.495275,
        40.698720,40.898259,41.093878,41.285560,41.473289,41.657052,41.836834,42.012621,
        42.184399,42.352155,42.515877,42.675552,42.831170,42.982718,43.130186,43.273565,
        43.412843,43.548012,43.679063,43.805987,43.928777,44.047425,44.161925,44.272269,
        44.378451,44.480467,44.578310,44.671978,44.761464,44.846766,44.927880,45.004805,
        45.077536,45.146074,45.210416,45.270562,45.326512,45.378265,45.425822,45.469185,
        45.508355,45.543334,45.574124,45.600730,45.623153,45.641399,45.655472,45.665376,
        45.671117,45.672701,45.670134,45.663424,45.652577,45.637601,45.618505,45.595297,
        45.567986,45.536582,45.501096,45.461537,45.417917,45.370248,45.318540,45.262808,
        45.203063,45.139319,45.071590,44.999890,44.924233,44.844636,44.761113,44.673681,
        44.582355,44.487154,44.388094,44.285193,44.178470,44.067943,43.953631,43.835554,
        43.713732,43.588186,43.458935,43.326002,43.189408,43.049175,42.905325,42.757883,
        42.606870,42.452311,42.294229,42.132650,41.967598,41.799099,41.627178,41.451862,
        41.273176,41.091148,40.905806,40.717176,40.525287,40.330167,40.131844,39.930349,
        39.725711,39.517958,39.307122,39.093234,38.876323,38.656421,38.433560,38.207771,
        37.979088,37.747542,37.513166,37.275994,37.036058,36.793394,36.548035,36.300015,
        36.049370,35.796134,35.540342,35.282031,35.021236,34.757994,34.492341,34.224313,
        33.953948,33.681284,33.406357,33.129205,32.849867,32.568381,32.284785,31.999118,
        31.711420,31.421729,31.130085,30.836528,30.541097,30.243833,29.944776,29.643967,
        29.341446,29.037255,28.731434,28.424024,28.115068,27.804607,27.492683,27.179338,
        26.864613,26.548552,26.231196,25.912589,25.592773,25.271791,24.949685,24.626500,
        24.302279,23.977064,23.650899,23.323829,22.995896,22.667144,22.337618,22.007361,
        21.676418,21.344832,21.012647,20.679909,20.346661,20.012947,19.678813,19.344303,
        19.009460,18.674330,18.338958,18.003388,17.667664,17.331831,16.995935,16.660019,
        16.324128,15.988307,15.652601,15.317054,14.981711,14.646616,14.311814,13.977350,
        13.643268,13.309612,12.976428,12.643758,12.311648,11.980142,11.649283,11.319117,
        10.989686,10.661035,10.333208,10.006247,9.680198,9.355103,9.031006,8.707950,
        8.385979,8.065134,7.745460,7.426998,7.109792,6.793884,6.479316,6.166130,
        5.854369,5.544075,5.235288,4.928050,4.622404,4.318389,4.016047,3.715419,
        3.416545,3.119465,2.824220,2.530850,2.239393,1.949891,1.662381,1.376904,
        1.093497,0.812200,0.533051,0.256087,-0.018653,-0.291132,-0.561312,-0.829157,
        -1.094630,-1.357694,-1.618313,-1.876451,-2.132072,-2.385142,-2.635624,-2.883485,
        -3.128689,-3.371203,-3.610993,-3.848025,-4.082267,-4.313684,-4.542245,-4.767917,
        -4.990669,-5.210468,-5.427284,-5.641086,-5.851843,-6.059525,-6.264102,-6.465545,
        -6.663824,-6.858911,-7.050778,-7.239396,-7.424737,-7.606775,-7.785483,-7.960834,
        -8.132801,-8.301360,-8.466485,-8.628151,-8.786333,-8.941008,-9.092151,-9.239739,
        -9.383749,-9.524160,-9.660948,-9.794091,-9.923570,-10.049362,-10.171448,-10.289807,
        -10.404421,-10.515268,-10.622332,-10.725594,-10.825036,-10.920640,-11.012390,-11.100269,
        -11.184261,-11.264350,-11.340522,-11.412761,-11.481054,-11.545386,-11.605744,-11.662115,
        -11.714487,-11.762848,-11.807187,-11.847491,-11.883751,-11.915957,-11.944098,-11.968166,
        -11.988152,-12.004048,-12.015846,-12.023538,-12.027118,-12.026579,-12.021916,-12.013123,
        -12.000195,-11.983127,-11.961916,-11.936558,-11.907050,-11.873390,-11.835575,-11.793604,
        -11.747476,-11.697189,-11.642744,-11.584142,-11.521382,-11.454467,-11.383397,-11.308175,
        -11.228804,-11.145288,-11.057628,-10.965831,-10.869899,-10.769839,-10.665656,-10.557356,
        -10.444945,-10.328430,-10.207820,-10.083121,-9.954342,-9.821492,-9.684580,-9.543617,
        -9.398611,-9.249575,-9.096519,-8.939454,-8.778394,-8.613349,-8.444335,-8.271363,
        -8.094448,-7.913605,-7.728847,-7.540191,-7.347652,-7.151246,-6.950991,-6.746902,
        -6.538997,-6.327296,-6.111815,-5.892573,-5.669591,-5.442887,-5.212482,-4.978396,
        -4.740650,-4.499266,-4.254265,-4.005670,-3.753503,-3.497787,-3.238546,-2.975804,
        -2.709584,-2.439912,-2.166813,-1.890312,-1.610434,-1.327207,-1.040656,-0.750810,
        -0.457694,-0.161337,0.138232,0.440986,0.746896,1.055932,1.368065,1.683264,
        2.001500,2.322742,2.646958,2.974118,3.304190,3.637141,3.972940,4.311554,
        4.652949,4.997092,5.343950,5.693489,6.045674,6.400471,6.757845,7.117761,
        7.480183,7.845076,8.212403,8.582128,8.954215,9.328627,9.705325,10.084274,
        10.465436,10.848772,11.234244,11.621814,12.011443,12.403092,12.796722,13.192294,
        13.589769,13.989105,14.390264,14.793204,15.197886,15.604269,16.012312,16.421973,
        16.833213,17.245988,17.660258,18.075981,18.493115,18.911617,19.331446,19.752559,
        20.174913,20.598466,21.023175,21.448997,21.875888,22.303806,22.732706,23.162546,
        23.593282,24.024870,24.457267,24.890427,25.324309,25.758866,26.194056,26.629833,
        27.066154,27.502974,27.940249,28.377933,28.815984,29.254355,29.693003,30.131883,
        30.570949,31.010158,31.449464,31.888822,32.328189,32.767519,33.206766,33.645888,
        34.084838,34.523572,34.962046,35.400214,35.838031,36.275454,36.712438,37.148937,
        37.584908,38.020307,38.455087,38.889206,39.322620,39.755283,40.187152,40.618182,
        41.048331,41.477553,41.905806,42.333046,42.759229,43.184311,43.608251,44.031003,
        44.452526,44.872777,45.291712,45.709290,46.125467,46.540202,46.953452,47.365176,
        47.775331,48.183876,48.590770,48.995971,49.399439,49.801132,50.201010,50.599032,
        50.995158,51.389348,51.781562,52.171761,52.559905,52.945955,53.329872,53.711618,
        54.091153,54.468441,54.843442,55.216120,55.586436,55.954354,56.319837,56.682848,
        57.043351,57.401310,57.756688,58.109452,58.459564,58.806991,59.151698,59.493651,
        59.832815,60.169158,60.502645,60.833244,61.160922,61.485647,61.807386,62.126108,
        62.441782,62.754376,63.063861,63.370205,63.673379,63.973352,64.270097,64.563584,
        64.853784,65.140670,65.424213,65.704387,65.981164,66.254517,66.524421,66.790849,
        67.053777,67.313178,67.569029,67.821305,68.069983,68.315038,68.556448,68.794190,
        69.028243,69.258583,69.485190,69.708043,69.927122,70.142405,70.353874,70.561510,
        70.765293,70.965205,71.161228,71.353345,71.541538,71.725791,71.906089,72.082414,
        72.254752,72.423088,72.587408,72.747698,72.903943,73.056132,73.204252,73.348290,
        73.488235,73.624076,73.755802,73.883403,74.006870,74.126191,74.241360,74.352368,
        74.459206,74.561867,74.660345,74.754632,74.844724,74.930613,75.012296,75.089768,
        75.163024,75.232061,75.296876,75.357466,75.413829,75.465963,75.513867,75.557540,
        75.596982,75.632193,75.663174,75.689925,75.712450,75.730748,75.744825,75.754681,
        75.760321,75.761749,75.758969,75.751987,75.740807,75.725436,75.705880,75.682145,
        75.654240,75.622172,75.585949,75.545580,75.501074,75.452441,75.399692,75.342835,
        75.281884,75.216848,75.147741,75.074575,74.997362,74.916116,74.830851,74.741581,
        74.648320,74.551085,74.449890,74.344752,74.235687,74.122713,74.005846,73.885104,
        73.760506,73.632070,73.499816,73.363763,73.223931,73.080341,72.933013,72.781969,
        72.627230,72.468819,72.306758,72.141070,71.971778,71.798907,71.622479,71.442521,
        71.259056,71.072111,70.881710,70.687880,70.490648,70.290041,70.086085,69.878809,
        69.668240,69.454406,69.237338,69.017063,68.793612,68.567013,68.337298,68.104497,
        67.868641,67.629761,67.387888,67.143055,66.895294,66.644637,66.391117,66.134768,
        65.875623,65.613716,65.349082,65.081753,64.811767,64.539157,64.263958,63.986208,
        63.705940,63.423193,63.138002,62.850403,62.560435,62.268134,61.973539,61.676686,
        61.377614,61.076361,60.772966,60.467468,60.159906,59.850318,59.538745,59.225226,
        58.909802,58.592512,58.273396,57.952496,57.629852,57.305505,56.979496,56.651867,
        56.322659,55.991913,55.659672,55.325978,54.990872,54.654397,54.316597,53.977512,
        53.637187,53.295663,52.952984,52.609194,52.264335,51.918451,51.571585,51.223782,
        50.875084,50.525536,50.175181,49.824064,49.472228,49.119718,48.766578,48.412853,
        48.058586,47.703822,47.348606,46.992981,46.636994,46.280687,45.924106,45.567296,
        45.210301,44.853166,44.495935,44.138653,43.781365,43.424116,43.066950,42.709912,
        42.353047,41.996399,41.640012,41.283932,40.928203,40.572869,40.217975,39.863565,
        39.509683,39.156374,38.803682,38.451651,38.100324,37.749747,37.399962,37.051013,
        36.702945,36.355800,36.009622,35.664454,35.320339,34.977321,34.635443,34.294746,
        33.955275,33.617071,33.280176,32.944633,32.610484,32.277771,31.946535,31.616818,
        31.288662,30.962107,30.637194,30.313964,29.992459,29.672717,29.354780,29.038686,
        28.724477,28.412192,28.101869,27.793548,27.487268,27.183067,26.880983,26.581056,
        26.283321,25.987818,25.694584,25.403655,25.115068,24.828860,24.545068,24.263726,
        23.984871,23.708538,23.434763,23.163579,22.895022,22.629125,22.365923,22.105449,
        21.847737,21.592818,21.340727,21.091494,20.845153,20.601734,20.361270,20.123790,
        19.889325,19.657907,19.429563,19.204325,18.982221,18.763279,18.547529,18.334999,
        18.125715,17.919706,17.716998,17.517618,17.321592,17.128945,16.939704,16.753893,
        16.571536,16.392658,16.217283,16.045433,15.877132,15.712403,15.551267,15.393747,
        15.239863,15.089637,14.943088,14.800237,14.661104,14.525707,14.394066,14.266198,
        14.142122,14.021855,13.905414,13.792815,13.684074,13.579208,13.478231,13.381159,
        13.288005,13.198783,13.113506,13.032188,12.954841,12.881478,12.812108,12.746745,
        12.685397,12.628077,12.574792,12.525552,12.480367,12.439244,12.402192,12.369216,
        12.340326,12.315526,12.294822,12.278222,12.265728,12.257346,12.253079,12.252932,
        12.256907,12.265007,12.277234,12.293590,12.314076,12.338692,12.367439,12.400317,
        12.437324,12.478459,12.523722,12.573109,12.626618,12.684246,12.745989,12.811844,
        12.881804,12.955867,13.034025,13.116274,13.202605,13.293014,13.387492,13.486031,
        13.588623,13.695260,13.805931,13.920627,14.039339,14.162054,14.288763,14.419453,
        14.554113,14.692730,14.835290,14.981781,15.132189,15.286498,15.444696,15.606765,
        15.772691,15.942457,16.116047,16.293443,16.474629,16.659586,16.848296,17.040740,
        17.236899,17.436753,17.640283,17.847467,18.058284,18.272715,18.490735,18.712325,
        18.937460,19.166117,19.398275,19.633907,19.872992,20.115502,20.361415,20.610703,
        20.863343,21.119306,21.378567,21.641098,21.906873,22.175864,22.448041,22.723378,
        23.001845,23.283413,23.568052,23.855733,24.146424,24.440097,24.736718,25.036258,
        25.338685,25.643966,25.952069,26.262962,26.576612,26.892985,27.212047,27.533765,
        27.858105,28.185032,28.514510,28.846506,29.180983,29.517906,29.857238,30.198945,
        30.542987,30.889330,31.237936,31.588767,31.941786,32.296955,32.654236,33.013590,
        33.374979,33.738364,34.103705,34.470964,34.840100,35.211075,35.583847,35.958378,
        36.334625,36.712550,37.092110,37.473265,37.855973,38.240194,38.625885,39.013005,
        39.401512,39.791364,40.182519,40.574934,40.968566,41.363373,41.759313,42.156341,
        42.554416,42.953493,43.353530,43.754482,44.156307,44.558961,44.962399,45.366579,
        45.771455,46.176985,46.583123,46.989826,47.397049,47.804748,48.212878,48.621396,
        49.030256,49.439414,49.848825,50.258444,50.668228,51.078131,51.488109,51.898116,
        52.308108,52.718041,53.127869,53.537548,53.947033,54.356279,54.765242,55.173877,
        55.582139,55.989983,56.397366,56.804242,57.210567,57.616297,58.021387,58.425794,
        58.829472,59.232377,59.634466,60.035695,60.436019,60.835396,61.233780,61.631130,
        62.027400,62.422548,62.816531,63.209305,63.600828,63.991056,64.379948,64.767460,
        65.153550,65.538175,65.921294,66.302865,66.682847,67.061196,67.437873,67.812835,
        68.186042,68.557453,68.927028,69.294725,69.660505,70.024328,70.386154,70.745943,
        71.103656,71.459254,71.812698,72.163949,72.512969,72.859720,73.204164,73.546263,
        73.885980,74.223278,74.558119,74.890468,75.220289,75.547544,75.872198,76.194216,
        76.513563,76.830204,77.144105,77.455230,77.763547,78.069021,78.371619,78.671309,
        78.968057,79.261832,79.552601,79.840333,80.124996,80.406560,80.684993,80.960266,
        81.232349,81.501211,81.766825,82.029160,82.288190,82.543884,82.796216,83.045159,
        83.290685,83.532769,83.771382,84.006501,84.238099,84.466151,84.690634,84.911522,
        85.128791,85.342419,85.552383,85.758659,85.961226,86.160061,86.355145,86.546454,
        86.733970,86.917672,87.097540,87.273556,87.445700,87.613953,87.778299,87.938720,
        88.095199,88.247718,88.396262,88.540816,88.681364,88.817891,88.950383,89.078826,
        89.203206,89.323511,89.439728,89.551846,89.659851,89.763734,89.863484,89.959090,
        90.050543,90.137834,90.220953,90.299892,90.374644,90.445201,90.511556,90.573703,
        90.631636,90.685349,90.734837,90.780096,90.821122,90.857911,90.890459,90.918765,
        90.942826,90.962640,90.978207,90.989525,90.996595,90.999416,90.997989,90.992316,
        90.982397,90.968236,90.949835,90.927196,90.900324,90.869223,90.833896,90.794350,
        90.750589,90.702620,90.650448,90.594081,90.533527,90.468792,90.399886,90.326817,
        90.249594,90.168227,90.082726,89.993102,89.899366,89.801530,89.699605,89.593605,
        89.483541,89.369428,89.251279,89.129108,89.002931,88.872763,88.738618,88.600514,
        88.458468,88.312494,88.162613,88.008840,87.851195,87.689697,87.524364,87.355216,
        87.182273,87.005556,86.825085,86.640882,86.452969,86.261368,86.066101,85.867191,
        85.664662,85.458537,85.248841,85.035598,84.818834,84.598573,84.374842,84.147666,
        83.917073,83.683088,83.445741,83.205057,82.961065,82.713794,82.463273,82.209530,
        81.952596,81.692500,81.429272,81.162942,80.893543,80.621105,80.345660,80.067239,
        79.785876,79.501602,79.214451,78.924456,78.631650,78.336068,78.037743,77.736711,
        77.433005,77.126662,76.817716,76.506203,76.192160,75.875622,75.556626,75.235209,
        74.911408,74.585260,74.256803,73.926075,73.593114,73.257958,72.920646,72.581217,
        72.239709,71.896163,71.550618,71.203113,70.853689,70.502386,70.149243,69.794303,
        69.437604,69.079190,68.719100,68.357376,67.994060,67.629194,67.262818,66.894976,
        66.525710,66.155061,65.783074,65.409789,65.035250,64.659500,64.282582,63.904540,
        63.525416,63.145254,62.764097,62.381990,61.998976,61.615099,61.230403,60.844931,
        60.458729,60.071839,59.684308,59.296178,58.907494,58.518301,58.128643,57.738565,
        57.348111,56.957326,56.566255,56.174942,55.783432,55.391770,55.000000,54.608167,
        54.216317,53.824493,53.432740,53.041104,52.649628,52.258358,51.867338,51.476612,
        51.086226,50.696223,50.306649,49.917547,49.528963,49.140939,48.753521,48.366752,
        47.980677,47.595339,47.210783,46.827052,46.444190,46.062240,45.681246,45.301251,
        44.922299,44.544432,44.167694,43.792127,43.417774,43.044677,42.672880,42.302423,
        41.933350,41.565701,41.199520,40.834847,40.471723,40.110191,39.750290,39.392063,
        39.035548,38.680788,38.327821,37.976689,37.627430,37.280085,36.934692,36.591291,
        36.249921,35.910620,35.573426,35.238379,34.905515,34.574872,34.246488,33.920400,
        33.596645,33.275258,32.956277,32.639738,32.325675,32.014125,31.705123,31.398704,
        31.094901,30.793750,30.495284,30.199537,29.906542,29.616332,29.328939,29.044397,
        28.762737,28.483990,28.208188,27.935362,27.665543,27.398760,27.135044,26.874424,
        26.616930,26.362589,26.111432,25.863485,25.618777,25.377334,25.139184,24.904354,
        24.672869,24.444755,24.220038,23.998743,23.780894,23.566515,23.355631,23.148265,
        22.944439,22.744177,22.547501,22.354431,22.164990,21.979199,21.797078,21.618646,
        21.443925,21.272932,21.105687,20.942208,20.782513,20.626619,20.474543,20.326303,
        20.181914,20.041391,19.904750,19.772006,19.643173,19.518266,19.397296,19.280278,
        19.167224,19.058146,18.953055,18.851963,18.754880,18.661817,18.572783,18.487787,
        18.406839,18.329946,18.257117,18.188359,18.123678,18.063082,18.006576,17.954166,
        17.905856,17.861651,17.821556,17.785574,17.753708,17.725960,17.702333,17.682829,
        17.667448,17.656192,17.649061,17.646053,17.647170,17.652409,17.661770,17.675248,
        17.692843,17.714552,17.740369,17.770292,17.804316,17.842436,17.884646,17.930941,
        17.981313,18.035757,18.094265,18.156829,18.223440,18.294091,18.368771,18.447471,
        18.530180,18.616889,18.707586,18.802259,18.900897,19.003486,19.110015,19.220469,
        19.334835,19.453099,19.575245,19.701258,19.831124,19.964825,20.102345,20.243667,
        20.388774,20.537648,20.690270,20.846622,21.006684,21.170438,21.337862,21.508937,
        21.683641,21.861954,22.043853,22.229317,22.418322,22.610846,22.806866,23.006357,
        23.209296,23.415657,23.625417,23.838549,24.055028,24.274827,24.497920,24.724281,
        24.953881,25.186693,25.422689,25.661841,25.904120,26.149496,26.397940,26.649422,
        26.903913,27.161381,27.421795,27.685124,27.951337,28.220401,28.492285,28.766955,
        29.044380,29.324524,29.607356,29.892840,30.180943,30.471631,30.764868,31.060619,
        31.358850,31.659523,31.962604,32.268056,32.575842,32.885926,33.198271,33.512839,
        33.829593,34.148494,34.469505,34.792587,35.117702,35.444811,35.773874,36.104854,
        36.437708,36.772400,37.108887,37.447130,37.787089,38.128723,38.471991,38.816853,
        39.163266,39.511190,39.860584,40.211404,40.563611,40.917161,41.272012,41.628122,
        41.985449,42.343949,42.703580,43.064300,43.426064,43.788830,44.152554,44.517194,
        44.882705,45.249044,45.616167,45.984030,46.352590,46.721801,47.091621,47.462005,
        47.832909,48.204288,48.576098,48.948295,49.320833,49.693670,50.066759,50.440057,
        50.813518,51.187099,51.560754,51.934439,52.308108,52.681718,53.055224,53.428580,
        53.801743,54.174667,54.547307,54.919620,55.291561,55.663084,56.034146,56.404702,
        56.774707,57.144118,57.512889,57.880977,58.248337,58.614926,58.980699,59.345612,
        59.709622,60.072684,60.434756,60.795793,61.155752,61.514590,61.872263,62.228730,
        62.583945,62.937868,63.290455,63.641663,63.991451,64.339776,64.686596,65.031869,
        65.375553,65.717608,66.057991,66.396661,66.733578,67.068701,67.401988,67.733401,
        68.062897,68.390438,68.715984,69.039496,69.360933,69.680257,69.997429,70.312410,
        70.625163,70.935648,71.243829,71.549667,71.853125,72.154166,72.452754,72.748851,
        73.042422,73.333430,73.621840,73.907617,74.190725,74.471130,74.748796,75.023691,
        75.295779,75.565028,75.831404,76.094874,76.355405,76.612966,76.867525,77.119049,
        77.367508,77.612870,77.855105,78.094183,78.330074,78.562748,78.792177,79.018331,
        79.241182,79.460702,79.676863,79.889638,80.099000,80.304923,80.507380,80.706346,
        80.901795,81.093702,81.282043,81.466793,81.647929,81.825427,81.999265,82.169420,
        82.335869,82.498591,82.657565,82.812770,82.964185,83.111790,83.255566,83.395494,
        83.531555,83.663730,83.792001,83.916352,84.036766,84.153225,84.265713,84.374216,
        84.478717,84.579203,84.675658,84.768069,84.856422,84.940705,85.020906,85.097011,
        85.169010,85.236891,85.300645,85.360259,85.415726,85.467036,85.514180,85.557149,
        85.595936,85.630534,85.660936,85.687134,85.709125,85.726901,85.740458,85.749791,
        85.754897,85.755771,85.752412,85.744815,85.732980,85.716904,85.696586,85.672025,
        85.643223,85.610177,85.572890,85.531363,85.485597,85.435594,85.381358,85.322890,
        85.260196,85.193279,85.122143,85.046793,84.967236,84.883477,84.795522,84.703379,
        84.607055,84.506557,84.401895,84.293076,84.180111,84.063009,83.941780,83.816434,
        83.686984,83.553440,83.415815,83.274122,83.128372,82.978580,82.824760,82.666926,
        82.505092,82.339274,82.169488,81.995750,81.818077,81.636485,81.450992,81.261615,
        81.068374,80.871287,80.670374,80.465653,80.257146,80.044871,79.828852,79.609108,
        79.385662,79.158536,78.927752,78.693333,78.455304,78.213687,77.968507,77.719789,
        77.467557,77.211838,76.952656,76.690038,76.424011,76.154602,75.881838,75.605747,
        75.326356,75.043695,74.757792,74.468677,74.176379,73.880927,73.582353,73.280687,
        72.975959,72.668202,72.357447,72.043726,71.727071,71.407514,71.085090,70.759831,
        70.431771,70.100943,69.767383,69.431124,69.092201,68.750650,68.406507,68.059806,
        67.710584,67.358877,67.004722,66.648155,66.289214,65.927937,65.564360,65.198522,
        64.830460,64.460215,64.087823,63.713324,63.336757,62.958161,62.577576,62.195042,
        61.810599,61.424287,61.036147,60.646218,60.254543,59.861162,59.466115,59.069446,
        58.671195,58.271404,57.870115,57.467370,57.063211,56.657681,56.250822,55.842678,
        55.433290,55.022702,54.610958,54.198099,53.784171,53.369215,52.953276,52.536398,
        52.118624,51.699998,51.280564,50.860366,50.439449,50.017857,49.595633,49.172823,
        48.749471,48.325621,47.901317,47.476605,47.051530,46.626134,46.200465,45.774565,
        45.348480,44.922255,44.495935,44.069563,43.643186,43.216847,42.790592,42.364465,
        41.938511,41.512774,41.087300,40.662133,40.237318,39.812899,39.388920,38.965426,
        38.542462,38.120072,37.698299,37.277189,36.856785,36.437131,36.018271,35.600249,
        35.183109,34.766894,34.351648,33.937414,33.524236,33.112156,32.701217,32.291463,
        31.882936,31.475679,31.069733,30.665142,30.261948,29.860192,29.459916,29.061162,
        28.663971,28.268384,27.874443,27.482189,27.091661,26.702901,26.315948,25.930844,
        25.547626,25.166336,24.787013,24.409695,24.034421,23.661231,23.290162,22.921254,
        22.554542,22.190067,21.827864,21.467971,21.110425,20.755262,20.402519,20.052232,
        19.704437,19.359168,19.016462,18.676352,18.338874,18.004062,17.671950,17.342571,
        17.015959,16.692147,16.371168,16.053053,15.737835,15.425545,15.116215,14.809876,
        14.506559,14.206293,13.909109,13.615036,13.324104,13.036341,12.751776,12.470437,
        12.192352,11.917548,11.646052,11.377891,11.113091,10.851677,10.593676,10.339112,
        10.088010,9.840394,9.596289,9.355716,9.118701,8.885264,8.655429,8.429216,
        8.206649,7.987746,7.772530,7.561020,7.353235,7.149196,6.948920,6.752426,
        6.559733,6.370857,6.185816,6.004625,5.827303,5.653863,5.484322,5.318695,
        5.156994,4.999236,4.845432,4.695596,4.549741,4.407879,4.270020,4.136177,
        4.006361,3.880580,3.758846,3.641167,3.527552,3.418010,3.312549,3.211175,
        3.113897,3.020719,2.931649,2.846692,2.765853,2.689136,2.616546,2.548086,
        2.483760,2.423570,2.367518,2.315606,2.267835,2.224206,2.184720,2.149376,
        2.118173,2.091110,2.068186,2.049399,2.034745,2.024223,2.017828,2.015556,
        2.017403,2.023365,2.033434,2.047606,2.065875,2.088233,2.114672,2.145187,
        2.179767,2.218404,2.261089,2.307812,2.358564,2.413334,2.472109,2.534880,
        2.601634,2.672359,2.747041,2.825667,2.908224,2.994697,3.085071,3.179332,
        3.277463,3.379449,3.485273,3.594918,3.708367,3.825602,3.946604,4.071355,
        4.199836,4.332027,4.467908,4.607459,4.750658,4.897485,5.047919,5.201935,
        5.359514,5.520630,5.685261,5.853384,6.024974,6.200006,6.378456,6.560298,
        6.745506,6.934055,7.125917,7.321067,7.519476,7.721117,7.925962,8.133982,
        8.345149,8.559434,8.776808,8.997239,9.220699,9.447156,9.676579,9.908939,
        10.144202,10.382336,10.623311,10.867093,11.113648,11.362945,11.614949,11.869627,
        12.126944,12.386866,12.649358,12.914385,13.181912,13.451903,13.724322,13.999132,
        14.276299,14.555783,14.837549,15.121560,15.407777,15.696163,15.986679,16.279288,
        16.573951,16.870629,17.169283,17.469873,17.772361,18.076706,18.382869,18.690809,
        19.000486,19.311859,19.624889,19.939533,20.255751,20.573501,20.892741,21.213431,
        21.535528,21.858991,22.183776,22.509842,22.837147,23.165647,23.495300,23.826063,
        24.157893,24.490746,24.824580,25.159351,25.495016,25.831531,26.168852,26.506936,
        26.845738,27.185214,27.525321,27.866014,28.207250,28.548983,28.891170,29.233766,
        29.576726,29.920007,30.263563,30.607350,30.951324,31.295439,31.639651,31.983916,
        32.328189,32.672425,33.016579,33.360607,33.704464,34.048105,34.391486,34.734562,
        35.077289,35.419622,35.761516,36.102928,36.443812,36.784125,37.123822,37.462858,
        37.801191,38.138774,38.475566,38.811521,39.146597,39.480748,39.813932,40.146105,
        40.477224,40.807245,41.136126,41.463823,41.790293,42.115494,42.439382,42.761916,
        43.083054,43.402752,43.720969,44.037663,44.352793,44.666317,44.978193,45.288381,
        45.596839,45.903527,46.208404,46.511431,46.812565,47.111769,47.409002,47.704224,
        47.997396,48.288480,48.577436,48.864227,49.148812,49.431155,49.711218,49.988963,
        50.264353,50.537350,50.807918,51.076021,51.341622,51.604686,51.865176,52.123057,
        52.378294,52.630853,52.880699,53.127798,53.372116,53.613619,53.852274,54.088049,
        54.320910,54.550826,54.777765,55.001694,55.222583,55.440401,55.655118,55.866702,
        56.075125,56.280357,56.482368,56.681130,56.876615,57.068794,57.257640,57.443125,
        57.625223,57.803908,57.979152,58.150931,58.319219,58.483991,58.645222,58.802888,
        58.956967,59.107433,59.254265,59.397439,59.536935,59.672729,59.804801,59.933130,
        60.057695,60.178477,60.295456,60.408613,60.517928,60.623385,60.724964,60.822649,
        60.916423,61.006269,61.092170,61.174113,61.252080,61.326058,61.396033,61.461990,
        61.523916,61.581798,61.635624,61.685382,61.731060,61.772647,61.810133,61.843508,
        61.872761,61.897884,61.918868,61.935704,61.948386,61.956905,61.961255,61.961429,
        61.957421,61.949227,61.936840,61.920257,61.899474,61.874487,61.845293,61.811889,
        61.774274,61.732445,61.686402,61.636144,61.581671,61.522983,61.460081,61.392965,
        61.321639,61.246104,61.166362,61.082418,60.994274,60.901934,60.805404,60.704689,
        60.599793,60.490723,60.377485,60.260087,60.138535,60.012838,59.883005,59.749043,
        59.610962,59.468773,59.322485,59.172108,59.017655,58.859137,58.696565,58.529953,
        58.359314,58.184660,58.006007,57.823367,57.636757,57.446191,57.251685,57.053255,
        56.850919,56.644692,56.434593,56.220640,56.002850,55.781243,55.555838,55.326654,
        55.093713,54.857033,54.616637,54.372545,54.124780,53.873364,53.618319,53.359668,
        53.097435,52.831643,52.562318,52.289483,52.013164,51.733386,51.450175,51.163557,
        50.873559,50.580208,50.283531,49.983556,49.680312,49.373826,49.064127,48.751245,
        48.435210,48.116051,47.793799,47.468484,47.140138,46.808791,46.474476,46.137224,
        45.797067,45.454040,45.108174,44.759502,44.408060,44.053880,43.696996,43.337444,
        42.975258,42.610474,42.243126,41.873251,41.500885,41.126064,40.748825,40.369204,
        39.987240,39.602968,39.216427,38.827655,38.436690,38.043571,37.648335,37.251023,
        36.851673,36.450324,36.047017,35.641791,35.234686,34.825742,34.415000,34.002501,
        33.588285,33.172394,32.754868,32.335750,31.915080,31.492901,31.069254,30.644183,
        30.217728,29.789932,29.360839,28.930490,28.498929,28.066199,27.632343,27.197403,
        26.761424,26.324449,25.886522,25.447686,25.007985,24.567464,24.126165,23.684134,
        23.241414,22.798050,22.354086,21.909567,21.464536,21.019039,20.573119,20.126823,
        19.680193,19.233276,18.786115,18.338756,17.891243,17.443621,16.995935,16.548229,
        16.100549,15.652939,15.205444,14.758108,14.310977,13.864096,13.417508,12.971259,
        12.525393,12.079955,11.634989,11.190540,10.746651,10.303369,9.860736,9.418797,
        8.977596,8.537178,8.097585,7.658862,7.221053,6.784201,6.348349,5.913542,
        5.479822,5.047233,4.615817,4.185617,3.756677,3.329039,2.902744,2.477837,
        2.054358,1.632349,1.211853,0.792911,0.375565,-0.040145,-0.454177,-0.866490,
        -1.277044,-1.685798,-2.092712,-2.497746,-2.900860,-3.302014,-3.701169,-4.098286,
        -4.493326,-4.886250,-5.277020,-5.665597,-6.051944,-6.436023,-6.817797,-7.197228,
        -7.574279,-7.948914,-8.321097,-8.690791,-9.057961,-9.422570,-9.784585,-10.143970,
        -10.500691,-10.854712,-11.206001,-11.554524,-11.900246,-12.243136,-12.583161,-12.920287,
        -13.254484,-13.585719,-13.913961,-14.239179,-14.561342,-14.880421,-15.196385,-15.509204,
        -15.818849,-16.125292,-16.428503,-16.728455,-17.025120,-17.318471,-17.608480,-17.895121,
        -18.178367,-18.458193,-18.734574,-19.007484,-19.276898,-19.542793,-19.805144,-20.063928,
        -20.319122,-20.570703,-20.818649,-21.062939,-21.303550,-21.540462,-21.773655,-22.003107,
        -22.228800,-22.450715,-22.668832,-22.883133,-23.093600,-23.300215,-23.502962,-23.701824,
        -23.896784,-24.087828,-24.274939,-24.458102,-24.637305,-24.812531,-24.983769,-25.151004,
        -25.314225,-25.473419,-25.628575,-25.779682,-25.926728,-26.069703,-26.208598,-26.343404,
        -26.474111,-26.600711,-26.723197,-26.841560,-26.955794,-27.065892,-27.171849,-27.273658,
        -27.371315,-27.464815,-27.554154,-27.639328,-27.720335,-27.797171,-27.869834,-27.938322,
        -28.002635,-28.062771,-28.118730,-28.170513,-28.218119,-28.261551,-28.300809,-28.335896,
        -28.366815,-28.393567,-28.416158,-28.434590,-28.448869,-28.458999,-28.464986,-28.466835,
        -28.464554,-28.458148,-28.447625,-28.432993,-28.414260,-28.391435,-28.364527,-28.333546,
        -28.298502,-28.259405,-28.216266,-28.169097,-28.117911,-28.062718,-28.003533,-27.940369,
        -27.873239,-27.802158,-27.727140,-27.648200,-27.565355,-27.478620,-27.388012,-27.293547,
        -27.195243,-27.093119,-26.987191,-26.877479,-26.764002,-26.646780,-26.525832,-26.401179,
        -26.272841,-26.140841,-26.005199,-25.865938,-25.723081,-25.576649,-25.426668,-25.273159,
        -25.116148,-24.955659,-24.791716,-24.624346,-24.453574,-24.279426,-24.101928,-23.921108,
        -23.736992,-23.549609,-23.358986,-23.165152,-22.968136,-22.767966,-22.564672,-22.358284,
        -22.148832,-21.936346,-21.720858,-21.502399,-21.281001,-21.056694,-20.829512,-20.599487,
        -20.366652,-20.131039,-19.892684,-19.651619,-19.407879,-19.161497,-18.912510,-18.660952,
        -18.406857,-18.150263,-17.891205,-17.629718,-17.365840,-17.099608,-16.831058,-16.560227,
        -16.287154,-16.011875,-15.734430,-15.454856,-15.173192,-14.889476,-14.603749,-14.316048,
        -14.026414,-13.734886,-13.441505,-13.146309,-12.849341,-12.550639,-12.250245,-11.948200,
        -11.644545,-11.339322,-11.032571,-10.724335,-10.414655,-10.103573,-9.791131,-9.477373,
        -9.162339,-8.846073,-8.528618,-8.210017,-7.890312,-7.569546,-7.247764,-6.925007,
        -6.601321,-6.276748,-5.951332,-5.625117,-5.298146,-4.970465,-4.642116,-4.313144,
        -3.983593,-3.653508,-3.322932,-2.991911,-2.660488,-2.328708,-1.996616,-1.664256,
        -1.331673,-0.998911,-0.666015,-0.333030,-0.000000,0.333030,0.666015,0.998911,
        1.331673,1.664256,1.996616,2.328708,2.660488,2.991911,3.322932,3.653508,
        3.983593,4.313144,4.642116,4.970465,5.298146,5.625117,5.951332,6.276748,
        6.601321,6.925007,7.247764,7.569546,7.890312,8.210017,8.528618,8.846073,
        9.162339,9.477373,9.791131,10.103573,10.414655,10.724335,11.032571,11.339322,
        11.644545,11.948200,12.250245,12.550639,12.849341,13.146309,13.441505,13.734886,
        14.026414,14.316048,14.603749,14.889476,15.173192,15.454856,15.734430,16.011875,
        16.287154,16.560227,16.831058,17.099608,17.365840,17.629718,17.891205,18.150263,
        18.406857,18.660952,18.912510,19.161497,19.407879,19.651619,19.892684,20.131039,
        20.366652,20.599487,20.829512,21.056694,21.281001,21.502399,21.720858,21.936346,
        22.148832,22.358284,22.564672,22.767966,22.968136,23.165152,23.358986,23.549609,
        23.736992,23.921108,24.101928,24.279426,24.453574,24.624346,24.791716,24.955659,
        25.116148,25.273159,25.426668,25.576649,25.723081,25.865938,26.005199,26.140841,
        26.272841,26.401179,26.525832,26.646780,26.764002,26.877479,26.987191,27.093119,
        27.195243,27.293547,27.388012,27.478620,27.565355,27.648200,27.727140,27.802158,
        27.873239,27.940369,28.003533,28.062718,28.117911,28.169097,28.216266,28.259405,
        28.298502,28.333546,28.364527,28.391435,28.414260,28.432993,28.447625,28.458148,
        28.464554,28.466835,28.464986,28.458999,28.448869,28.434590,28.416158,28.393567,
        28.366815,28.335896,28.300809,28.261551,28.218119,28.170513,28.118730,28.062771,
        28.002635,27.938322,27.869834,27.797171,27.720335,27.639328,27.554154,27.464815,
        27.371315,27.273658,27.171849,27.065892,26.955794,26.841560,26.723197,26.600711,
        26.474111,26.343404,26.208598,26.069703,25.926728,25.779682,25.628575,25.473419,
        25.314225,25.151004,24.983769,24.812531,24.637305,24.458102,24.274939,24.087828,
        23.896784,23.701824,23.502962,23.300215,23.093600,22.883133,22.668832,22.450715,
        22.228800,22.003107,21.773655,21.540462,21.303550,21.062939,20.818649,20.570703,
        20.319122,20.063928,19.805144,19.542793,19.276898,19.007484,18.734574,18.458193,
        18.178367,17.895121,17.608480,17.318471,17.025120,16.728455,16.428503,16.125292,
        15.818849,15.509204,15.196385,14.880421,14.561342,14.239179,13.913961,13.585719,
        13.254484,12.920287,12.583161,12.243136,11.900246,11.554524,11.206001,10.854712,
        10.500691,10.143970,9.784585,9.422570,9.057961,8.690791,8.321097,7.948914,
        7.574279,7.197228,6.817797,6.436023,6.051944,5.665597,5.277020,4.886250,
        4.493326,4.098286,3.701169,3.302014,2.900860,2.497746,2.092712,1.685798,
        1.277044,0.866490,0.454177,0.040145,-0.375565,-0.792911,-1.211853,-1.632349,
        -2.054358,-2.477837,-2.902744,-3.329039,-3.756677,-4.185617,-4.615817,-5.047233,
        -5.479822,-5.913542,-6.348349,-6.784201,-7.221053,-7.658862,-8.097585,-8.537178,
        -8.977596,-9.418797,-9.860736,-10.303369,-10.746651,-11.190540,-11.634989,-12.079955,
        -12.525393,-12.971259,-13.417508,-13.864096,-14.310977,-14.758108,-15.205444,-15.652939,
        -16.100549,-16.548229,-16.995935,-17.443621,-17.891243,-18.338756,-18.786115,-19.233276,
        -19.680193,-20.126823,-20.573119,-21.019039,-21.464536,-21.909567,-22.354086,-22.798050,
        -23.241414,-23.684134,-24.126165,-24.567464,-25.007985,-25.447686,-25.886522,-26.324449,
        -26.761424,-27.197403,-27.632343,-28.066199,-28.498929,-28.930490,-29.360839,-29.789932,
        -30.217728,-30.644183,-31.069254,-31.492901,-31.915080,-32.335750,-32.754868,-33.172394,
        -33.588285,-34.002501,-34.415000,-34.825742,-35.234686,-35.641791,-36.047017,-36.450324,
        -36.851673,-37.251023,-37.648335,-38.043571,-38.436690,-38.827655,-39.216427,-39.602968,
        -39.987240,-40.369204,-40.748825,-41.126064,-41.500885,-41.873251,-42.243126,-42.610474,
        -42.975258,-43.337444,-43.696996,-44.053880,-44.408060,-44.759502,-45.108174,-45.454040,
        -45.797067,-46.137224,-46.474476,-46.808791,-47.140138,-47.468484,-47.793799,-48.116051,
        -48.435210,-48.751245,-49.064127,-49.373826,-49.680312,-49.983556,-50.283531,-50.580208,
        -50.873559,-51.163557,-51.450175,-51.733386,-52.013164,-52.289483,-52.562318,-52.831643,
        -53.097435,-53.359668,-53.618319,-53.873364,-54.124780,-54.372545,-54.616637,-54.857033,
        -55.093713,-55.326654,-55.555838,-55.781243,-56.002850,-56.220640,-56.434593,-56.644692,
        -56.850919,-57.053255,-57.251685,-57.446191,-57.636757,-57.823367,-58.006007,-58.184660,
        -58.359314,-58.529953,-58.696565,-58.859137,-59.017655,-59.172108,-59.322485,-59.468773,
        -59.610962,-59.749043,-59.883005,-60.012838,-60.138535,-60.260087,-60.377485,-60.490723,
        -60.599793,-60.704689,-60.805404,-60.901934,-60.994274,-61.082418,-61.166362,-61.246104,
        -61.321639,-61.392965,-61.460081,-61.522983,-61.581671,-61.636144,-61.686402,-61.732445,
        -61.774274,-61.811889,-61.845293,-61.874487,-61.899474,-61.920257,-61.936840,-61.949227,
        -61.957421,-61.961429,-61.961255,-61.956905,-61.948386,-61.935704,-61.918868,-61.897884,
        -61.872761,-61.843508,-61.810133,-61.772647,-61.731060,-61.685382,-61.635624,-61.581798,
        -61.523916,-61.461990,-61.396033,-61.326058,-61.252080,-61.174113,-61.092170,-61.006269,
        -60.916423,-60.822649,-60.724964,-60.623385,-60.517928,-60.408613,-60.295456,-60.178477,
        -60.057695,-59.933130,-59.804801,-59.672729,-59.536935,-59.397439,-59.254265,-59.107433,
        -58.956967,-58.802888,-58.645222,-58.483991,-58.319219,-58.150931,-57.979152,-57.803908,
        -57.625223,-57.443125,-57.257640,-57.068794,-56.876615,-56.681130,-56.482368,-56.280357,
        -56.075125,-55.866702,-55.655118,-55.440401,-55.222583,-55.001694,-54.777765,-54.550826,
        -54.320910,-54.088049,-53.852274,-53.613619,-53.372116,-53.127798,-52.880699,-52.630853,
        -52.378294,-52.123057,-51.865176,-51.604686,-51.341622,-51.076021,-50.807918,-50.537350,
        -50.264353,-49.988963,-49.711218,-49.431155,-49.148812,-48.864227,-48.577436,-48.288480,
        -47.997396,-47.704224,-47.409002,-47.111769,-46.812565,-46.511431,-46.208404,-45.903527,
        -45.596839,-45.288381,-44.978193,-44.666317,-44.352793,-44.037663,-43.720969,-43.402752,
        -43.083054,-42.761916,-42.439382,-42.115494,-41.790293,-41.463823,-41.136126,-40.807245,
        -40.477224,-40.146105,-39.813932,-39.480748,-39.146597,-38.811521,-38.475566,-38.138774,
        -37.801191,-37.462858,-37.123822,-36.784125,-36.443812,-36.102928,-35.761516,-35.419622,
        -35.077289,-34.734562,-34.391486,-34.048105,-33.704464,-33.360607,-33.016579,-32.672425,
        -32.328189,-31.983916,-31.639651,-31.295439,-30.951324,-30.607350,-30.263563,-29.920007,
        -29.576726,-29.233766,-28.891170,-28.548983,-28.207250,-27.866014,-27.525321,-27.185214,
        -26.845738,-26.506936,-26.168852,-25.831531,-25.495016,-25.159351,-24.824580,-24.490746,
        -24.157893,-23.826063,-23.495300,-23.165647,-22.837147,-22.509842,-22.183776,-21.858991,
        -21.535528,-21.213431,-20.892741,-20.573501,-20.255751,-19.939533,-19.624889,-19.311859,
        -19.000486,-18.690809,-18.382869,-18.076706,-17.772361,-17.469873,-17.169283,-16.870629,
        -16.573951,-16.279288,-15.986679,-15.696163,-15.407777,-15.121560,-14.837549,-14.555783,
        -14.276299,-13.999132,-13.724322,-13.451903,-13.181912,-12.914385,-12.649358,-12.386866,
        -12.126944,-11.869627,-11.614949,-11.362945,-11.113648,-10.867093,-10.623311,-10.382336,
        -10.144202,-9.908939,-9.676579,-9.447156,-9.220699,-8.997239,-8.776808,-8.559434,
        -8.345149,-8.133982,-7.925962,-7.721117,-7.519476,-7.321067,-7.125917,-6.934055,
        -6.745506,-6.560298,-6.378456,-6.200006,-6.024974,-5.853384,-5.685261,-5.520630,
        -5.359514,-5.201935,-5.047919,-4.897485,-4.750658,-4.607459,-4.467908,-4.332027,
        -4.199836,-4.071355,-3.946604,-3.825602,-3.708367,-3.594918,-3.485273,-3.379449,
        -3.277463,-3.179332,-3.085071,-2.994697,-2.908224,-2.825667,-2.747041,-2.672359,
        -2.601634,-2.534880,-2.472109,-2.413334,-2.358564,-2.307812,-2.261089,-2.218404,
        -2.179767,-2.145187,-2.114672,-2.088233,-2.065875,-2.047606,-2.033434,-2.023365,
        -2.017403,-2.015556,-2.017828,-2.024223,-2.034745,-2.049399,-2.068186,-2.091110,
        -2.118173,-2.149376,-2.184720,-2.224206,-2.267835,-2.315606,-2.367518,-2.423570,
        -2.483760,-2.548086,-2.616546,-2.689136,-2.765853,-2.846692,-2.931649,-3.020719,
        -3.113897,-3.211175,-3.312549,-3.418010,-3.527552,-3.641167,-3.758846,-3.880580,
        -4.006361,-4.136177,-4.270020,-4.407879,-4.549741,-4.695596,-4.845432,-4.999236,
        -5.156994,-5.318695,-5.484322,-5.653863,-5.827303,-6.004625,-6.185816,-6.370857,
        -6.559733,-6.752426,-6.948920,-7.149196,-7.353235,-7.561020,-7.772530,-7.987746,
        -8.206649,-8.429216,-8.655429,-8.885264,-9.118701,-9.355716,-9.596289,-9.840394,
        -10.088010,-10.339112,-10.593676,-10.851677,-11.113091,-11.377891,-11.646052,-11.917548,
        -12.192352,-12.470437,-12.751776,-13.036341,-13.324104,-13.615036,-13.909109,-14.206293,
        -14.506559,-14.809876,-15.116215,-15.425545,-15.737835,-16.053053,-16.371168,-16.692147,
        -17.015959,-17.342571,-17.671950,-18.004062,-18.338874,-18.676352,-19.016462,-19.359168,
        -19.704437,-20.052232,-20.402519,-20.755262,-21.110425,-21.467971,-21.827864,-22.190067,
        -22.554542,-22.921254,-23.290162,-23.661231,-24.034421,-24.409695,-24.787013,-25.166336,
        -25.547626,-25.930844,-26.315948,-26.702901,-27.091661,-27.482189,-27.874443,-28.268384,
        -28.663971,-29.061162,-29.459916,-29.860192,-30.261948,-30.665142,-31.069733,-31.475679,
        -31.882936,-32.291463,-32.701217,-33.112156,-33.524236,-33.937414,-34.351648,-34.766894,
        -35.183109,-35.600249,-36.018271,-36.437131,-36.856785,-37.277189,-37.698299,-38.120072,
        -38.542462,-38.965426,-39.388920,-39.812899,-40.237318,-40.662133,-41.087300,-41.512774,
        -41.938511,-42.364465,-42.790592,-43.216847,-43.643186,-44.069563,-44.495935,-44.922255,
        -45.348480,-45.774565,-46.200465,-46.626134,-47.051530,-47.476605,-47.901317,-48.325621,
        -48.749471,-49.172823,-49.595633,-50.017857,-50.439449,-50.860366,-51.280564,-51.699998,
        -52.118624,-52.536398,-52.953276,-53.369215,-53.784171,-54.198099,-54.610958,-55.022702,
        -55.433290,-55.842678,-56.250822,-56.657681,-57.063211,-57.467370,-57.870115,-58.271404,
        -58.671195,-59.069446,-59.466115,-59.861162,-60.254543,-60.646218,-61.036147,-61.424287,
        -61.810599,-62.195042,-62.577576,-62.958161,-63.336757,-63.713324,-64.087823,-64.460215,
        -64.830460,-65.198522,-65.564360,-65.927937,-66.289214,-66.648155,-67.004722,-67.358877,
        -67.710584,-68.059806,-68.406507,-68.750650,-69.092201,-69.431124,-69.767383,-70.100943,
        -70.431771,-70.759831,-71.085090,-71.407514,-71.727071,-72.043726,-72.357447,-72.668202,
        -72.975959,-73.280687,-73.582353,-73.880927,-74.176379,-74.468677,-74.757792,-75.043695,
        -75.326356,-75.605747,-75.881838,-76.154602,-76.424011,-76.690038,-76.952656,-77.211838,
        -77.467557,-77.719789,-77.968507,-78.213687,-78.455304,-78.693333,-78.927752,-79.158536,
        -79.385662,-79.609108,-79.828852,-80.044871,-80.257146,-80.465653,-80.670374,-80.871287,
        -81.068374,-81.261615,-81.450992,-81.636485,-81.818077,-81.995750,-82.169488,-82.339274,
        -82.505092,-82.666926,-82.824760,-82.978580,-83.128372,-83.274122,-83.415815,-83.553440,
        -83.686984,-83.816434,-83.941780,-84.063009,-84.180111,-84.293076,-84.401895,-84.506557,
        -84.607055,-84.703379,-84.795522,-84.883477,-84.967236,-85.046793,-85.122143,-85.193279,
        -85.260196,-85.322890,-85.381358,-85.435594,-85.485597,-85.531363,-85.572890,-85.610177,
        -85.643223,-85.672025,-85.696586,-85.716904,-85.732980,-85.744815,-85.752412,-85.755771,
        -85.754897,-85.749791,-85.740458,-85.726901,-85.709125,-85.687134,-85.660936,-85.630534,
        -85.595936,-85.557149,-85.514180,-85.467036,-85.415726,-85.360259,-85.300645,-85.236891,
        -85.169010,-85.097011,-85.020906,-84.940705,-84.856422,-84.768069,-84.675658,-84.579203,
        -84.478717,-84.374216,-84.265713,-84.153225,-84.036766,-83.916352,-83.792001,-83.663730,
        -83.531555,-83.395494,-83.255566,-83.111790,-82.964185,-82.812770,-82.657565,-82.498591,
        -82.335869,-82.169420,-81.999265,-81.825427,-81.647929,-81.466793,-81.282043,-81.093702,
        -80.901795,-80.706346,-80.507380,-80.304923,-80.099000,-79.889638,-79.676863,-79.460702,
        -79.241182,-79.018331,-78.792177,-78.562748,-78.330074,-78.094183,-77.855105,-77.612870,
        -77.367508,-77.119049,-76.867525,-76.612966,-76.355405,-76.094874,-75.831404,-75.565028,
        -75.295779,-75.023691,-74.748796,-74.471130,-74.190725,-73.907617,-73.621840,-73.333430,
        -73.042422,-72.748851,-72.452754,-72.154166,-71.853125,-71.549667,-71.243829,-70.935648,
        -70.625163,-70.312410,-69.997429,-69.680257,-69.360933,-69.039496,-68.715984,-68.390438,
        -68.062897,-67.733401,-67.401988,-67.068701,-66.733578,-66.396661,-66.057991,-65.717608,
        -65.375553,-65.031869,-64.686596,-64.339776,-63.991451,-63.641663,-63.290455,-62.937868,
        -62.583945,-62.228730,-61.872263,-61.514590,-61.155752,-60.795793,-60.434756,-60.072684,
        -59.709622,-59.345612,-58.980699,-58.614926,-58.248337,-57.880977,-57.512889,-57.144118,
        -56.774707,-56.404702,-56.034146,-55.663084,-55.291561,-54.919620,-54.547307,-54.174667,
        -53.801743,-53.428580,-53.055224,-52.681718,-52.308108,-51.934439,-51.560754,-51.187099,
        -50.813518,-50.440057,-50.066759,-49.693670,-49.320833,-48.948295,-48.576098,-48.204288,
        -47.832909,-47.462005,-47.091621,-46.721801,-46.352590,-45.984030,-45.616167,-45.249044,
        -44.882705,-44.517194,-44.152554,-43.788830,-43.426064,-43.064300,-42.703580,-42.343949,
        -41.985449,-41.628122,-41.272012,-40.917161,-40.563611,-40.211404,-39.860584,-39.511190,
        -39.163266,-38.816853,-38.471991,-38.128723,-37.787089,-37.447130,-37.108887,-36.772400,
        -36.437708,-36.104854,-35.773874,-35.444811,-35.117702,-34.792587,-34.469505,-34.148494,
        -33.829593,-33.512839,-33.198271,-32.885926,-32.575842,-32.268056,-31.962604,-31.659523,
        -31.358850,-31.060619,-30.764868,-30.471631,-30.180943,-29.892840,-29.607356,-29.324524,
        -29.044380,-28.766955,-28.492285,-28.220401,-27.951337,-27.685124,-27.421795,-27.161381,
        -26.903913,-26.649422,-26.397940,-26.149496,-25.904120,-25.661841,-25.422689,-25.186693,
        -24.953881,-24.724281,-24.497920,-24.274827,-24.055028,-23.838549,-23.625417,-23.415657,
        -23.209296,-23.006357,-22.806866,-22.610846,-22.418322,-22.229317,-22.043853,-21.861954,
        -21.683641,-21.508937,-21.337862,-21.170438,-21.006684,-20.846622,-20.690270,-20.537648,
        -20.388774,-20.243667,-20.102345,-19.964825,-19.831124,-19.701258,-19.575245,-19.453099,
        -19.334835,-19.220469,-19.110015,-19.003486,-18.900897,-18.802259,-18.707586,-18.616889,
        -18.530180,-18.447471,-18.368771,-18.294091,-18.223440,-18.156829,-18.094265,-18.035757,
        -17.981313,-17.930941,-17.884646,-17.842436,-17.804316,-17.770292,-17.740369,-17.714552,
        -17.692843,-17.675248,-17.661770,-17.652409,-17.647170,-17.646053,-17.649061,-17.656192,
        -17.667448,-17.682829,-17.702333,-17.725960,-17.753708,-17.785574,-17.821556,-17.861651,
        -17.905856,-17.954166,-18.006576,-18.063082,-18.123678,-18.188359,-18.257117,-18.329946,
        -18.406839,-18.487787,-18.572783,-18.661817,-18.754880,-18.851963,-18.953055,-19.058146,
        -19.167224,-19.280278,-19.397296,-19.518266,-19.643173,-19.772006,-19.904750,-20.041391,
        -20.181914,-20.326303,-20.474543,-20.626619,-20.782513,-20.942208,-21.105687,-21.272932,
        -21.443925,-21.618646,-21.797078,-21.979199,-22.164990,-22.354431,-22.547501,-22.744177,
        -22.944439,-23.148265,-23.355631,-23.566515,-23.780894,-23.998743,-24.220038,-24.444755,
        -24.672869,-24.904354,-25.139184,-25.377334,-25.618777,-25.863485,-26.111432,-26.362589,
        -26.616930,-26.874424,-27.135044,-27.398760,-27.665543,-27.935362,-28.208188,-28.483990,
        -28.762737,-29.044397,-29.328939,-29.616332,-29.906542,-30.199537,-30.495284,-30.793750,
        -31.094901,-31.398704,-31.705123,-32.014125,-32.325675,-32.639738,-32.956277,-33.275258,
        -33.596645,-33.920400,-34.246488,-34.574872,-34.905515,-35.238379,-35.573426,-35.910620,
        -36.249921,-36.591291,-36.934692,-37.280085,-37.627430,-37.976689,-38.327821,-38.680788,
        -39.035548,-39.392063,-39.750290,-40.110191,-40.471723,-40.834847,-41.199520,-41.565701,
        -41.933350,-42.302423,-42.672880,-43.044677,-43.417774,-43.792127,-44.167694,-44.544432,
        -44.922299,-45.301251,-45.681246,-46.062240,-46.444190,-46.827052,-47.210783,-47.595339,
        -47.980677,-48.366752,-48.753521,-49.140939,-49.528963,-49.917547,-50.306649,-50.696223,
        -51.086226,-51.476612,-51.867338,-52.258358,-52.649628,-53.041104,-53.432740,-53.824493,
        -54.216317,-54.608167,-55.000000,-55.391770,-55.783432,-56.174942,-56.566255,-56.957326,
        -57.348111,-57.738565,-58.128643,-58.518301,-58.907494,-59.296178,-59.684308,-60.071839,
        -60.458729,-60.844931,-61.230403,-61.615099,-61.998976,-62.381990,-62.764097,-63.145254,
        -63.525416,-63.904540,-64.282582,-64.659500,-65.035250,-65.409789,-65.783074,-66.155061,
        -66.525710,-66.894976,-67.262818,-67.629194,-67.994060,-68.357376,-68.719100,-69.079190,
        -69.437604,-69.794303,-70.149243,-70.502386,-70.853689,-71.203113,-71.550618,-71.896163,
        -72.239709,-72.581217,-72.920646,-73.257958,-73.593114,-73.926075,-74.256803,-74.585260,
        -74.911408,-75.235209,-75.556626,-75.875622,-76.192160,-76.506203,-76.817716,-77.126662,
        -77.433005,-77.736711,-78.037743,-78.336068,-78.631650,-78.924456,-79.214451,-79.501602,
        -79.785876,-80.067239,-80.345660,-80.621105,-80.893543,-81.162942,-81.429272,-81.692500,
        -81.952596,-82.209530,-82.463273,-82.713794,-82.961065,-83.205057,-83.445741,-83.683088,
        -83.917073,-84.147666,-84.374842,-84.598573,-84.818834,-85.035598,-85.248841,-85.458537,
        -85.664662,-85.867191,-86.066101,-86.261368,-86.452969,-86.640882,-86.825085,-87.005556,
        -87.182273,-87.355216,-87.524364,-87.689697,-87.851195,-88.008840,-88.162613,-88.312494,
        -88.458468,-88.600514,-88.738618,-88.872763,-89.002931,-89.129108,-89.251279,-89.369428,
        -89.483541,-89.593605,-89.699605,-89.801530,-89.899366,-89.993102,-90.082726,-90.168227,
        -90.249594,-90.326817,-90.399886,-90.468792,-90.533527,-90.594081,-90.650448,-90.702620,
        -90.750589,-90.794350,-90.833896,-90.869223,-90.900324,-90.927196,-90.949835,-90.968236,
        -90.982397,-90.992316,-90.997989,-90.999416,-90.996595,-90.989525,-90.978207,-90.962640,
        -90.942826,-90.918765,-90.890459,-90.857911,-90.821122,-90.780096,-90.734837,-90.685349,
        -90.631636,-90.573703,-90.511556,-90.445201,-90.374644,-90.299892,-90.220953,-90.137834,
        -90.050543,-89.959090,-89.863484,-89.763734,-89.659851,-89.551846,-89.439728,-89.323511,
        -89.203206,-89.078826,-88.950383,-88.817891,-88.681364,-88.540816,-88.396262,-88.247718,
        -88.095199,-87.938720,-87.778299,-87.613953,-87.445700,-87.273556,-87.097540,-86.917672,
        -86.733970,-86.546454,-86.355145,-86.160061,-85.961226,-85.758659,-85.552383,-85.342419,
        -85.128791,-84.911522,-84.690634,-84.466151,-84.238099,-84.006501,-83.771382,-83.532769,
        -83.290685,-83.045159,-82.796216,-82.543884,-82.288190,-82.029160,-81.766825,-81.501211,
        -81.232349,-80.960266,-80.684993,-80.406560,-80.124996,-79.840333,-79.552601,-79.261832,
        -78.968057,-78.671309,-78.371619,-78.069021,-77.763547,-77.455230,-77.144105,-76.830204,
        -76.513563,-76.194216,-75.872198,-75.547544,-75.220289,-74.890468,-74.558119,-74.223278,
        -73.885980,-73.546263,-73.204164,-72.859720,-72.512969,-72.163949,-71.812698,-71.459254,
        -71.103656,-70.745943,-70.386154,-70.024328,-69.660505,-69.294725,-68.927028,-68.557453,
        -68.186042,-67.812835,-67.437873,-67.061196,-66.682847,-66.302865,-65.921294,-65.538175,
        -65.153550,-64.767460,-64.379948,-63.991056,-63.600828,-63.209305,-62.816531,-62.422548,
        -62.027400,-61.631130,-61.233780,-60.835396,-60.436019,-60.035695,-59.634466,-59.232377,
        -58.829472,-58.425794,-58.021387,-57.616297,-57.210567,-56.804242,-56.397366,-55.989983,
        -55.582139,-55.173877,-54.765242,-54.356279,-53.947033,-53.537548,-53.127869,-52.718041,
        -52.308108,-51.898116,-51.488109,-51.078131,-50.668228,-50.258444,-49.848825,-49.439414,
        -49.030256,-48.621396,-48.212878,-47.804748,-47.397049,-46.989826,-46.583123,-46.176985,
        -45.771455,-45.366579,-44.962399,-44.558961,-44.156307,-43.754482,-43.353530,-42.953493,
        -42.554416,-42.156341,-41.759313,-41.363373,-40.968566,-40.574934,-40.182519,-39.791364,
        -39.401512,-39.013005,-38.625885,-38.240194,-37.855973,-37.473265,-37.092110,-36.712550,
        -36.334625,-35.958378,-35.583847,-35.211075,-34.840100,-34.470964,-34.103705,-33.738364,
        -33.374979,-33.013590,-32.654236,-32.296955,-31.941786,-31.588767,-31.237936,-30.889330,
        -30.542987,-30.198945,-29.857238,-29.517906,-29.180983,-28.846506,-28.514510,-28.185032,
        -27.858105,-27.533765,-27.212047,-26.892985,-26.576612,-26.262962,-25.952069,-25.643966,
        -25.338685,-25.036258,-24.736718,-24.440097,-24.146424,-23.855733,-23.568052,-23.283413,
        -23.001845,-22.723378,-22.448041,-22.175864,-21.906873,-21.641098,-21.378567,-21.119306,
        -20.863343,-20.610703,-20.361415,-20.115502,-19.872992,-19.633907,-19.398275,-19.166117,
        -18.937460,-18.712325,-18.490735,-18.272715,-18.058284,-17.847467,-17.640283,-17.436753,
        -17.236899,-17.040740,-16.848296,-16.659586,-16.474629,-16.293443,-16.116047,-15.942457,
        -15.772691,-15.606765,-15.444696,-15.286498,-15.132189,-14.981781,-14.835290,-14.692730,
        -14.554113,-14.419453,-14.288763,-14.162054,-14.039339,-13.920627,-13.805931,-13.695260,
        -13.588623,-13.486031,-13.387492,-13.293014,-13.202605,-13.116274,-13.034025,-12.955867,
        -12.881804,-12.811844,-12.745989,-12.684246,-12.626618,-12.573109,-12.523722,-12.478459,
        -12.437324,-12.400317,-12.367439,-12.338692,-12.314076,-12.293590,-12.277234,-12.265007,
        -12.256907,-12.252932,-12.253079,-12.257346,-12.265728,-12.278222,-12.294822,-12.315526,
        -12.340326,-12.369216,-12.402192,-12.439244,-12.480367,-12.525552,-12.574792,-12.628077,
        -12.685397,-12.746745,-12.812108,-12.881478,-12.954841,-13.032188,-13.113506,-13.198783,
        -13.288005,-13.381159,-13.478231,-13.579208,-13.684074,-13.792815,-13.905414,-14.021855,
        -14.142122,-14.266198,-14.394066,-14.525707,-14.661104,-14.800237,-14.943088,-15.089637,
        -15.239863,-15.393747,-15.551267,-15.712403,-15.877132,-16.045433,-16.217283,-16.392658,
        -16.571536,-16.753893,-16.939704,-17.128945,-17.321592,-17.517618,-17.716998,-17.919706,
        -18.125715,-18.334999,-18.547529,-18.763279,-18.982221,-19.204325,-19.429563,-19.657907,
        -19.889325,-20.123790,-20.361270,-20.601734,-20.845153,-21.091494,-21.340727,-21.592818,
        -21.847737,-22.105449,-22.365923,-22.629125,-22.895022,-23.163579,-23.434763,-23.708538,
        -23.984871,-24.263726,-24.545068,-24.828860,-25.115068,-25.403655,-25.694584,-25.987818,
        -26.283321,-26.581056,-26.880983,-27.183067,-27.487268,-27.793548,-28.101869,-28.412192,
        -28.724477,-29.038686,-29.354780,-29.672717,-29.992459,-30.313964,-30.637194,-30.962107,
        -31.288662,-31.616818,-31.946535,-32.277771,-32.610484,-32.944633,-33.280176,-33.617071,
        -33.955275,-34.294746,-34.635443,-34.977321,-35.320339,-35.664454,-36.009622,-36.355800,
        -36.702945,-37.051013,-37.399962,-37.749747,-38.100324,-38.451651,-38.803682,-39.156374,
        -39.509683,-39.863565,-40.217975,-40.572869,-40.928203,-41.283932,-41.640012,-41.996399,
        -42.353047,-42.709912,-43.066950,-43.424116,-43.781365,-44.138653,-44.495935,-44.853166,
        -45.210301,-45.567296,-45.924106,-46.280687,-46.636994,-46.992981,-47.348606,-47.703822,
        -48.058586,-48.412853,-48.766578,-49.119718,-49.472228,-49.824064,-50.175181,-50.525536,
        -50.875084,-51.223782,-51.571585,-51.918451,-52.264335,-52.609194,-52.952984,-53.295663,
        -53.637187,-53.977512,-54.316597,-54.654397,-54.990872,-55.325978,-55.659672,-55.991913,
        -56.322659,-56.651867,-56.979496,-57.305505,-57.629852,-57.952496,-58.273396,-58.592512,
        -58.909802,-59.225226,-59.538745,-59.850318,-60.159906,-60.467468,-60.772966,-61.076361,
        -61.377614,-61.676686,-61.973539,-62.268134,-62.560435,-62.850403,-63.138002,-63.423193,
        -63.705940,-63.986208,-64.263958,-64.539157,-64.811767,-65.081753,-65.349082,-65.613716,
        -65.875623,-66.134768,-66.391117,-66.644637,-66.895294,-67.143055,-67.387888,-67.629761,
        -67.868641,-68.104497,-68.337298,-68.567013,-68.793612,-69.017063,-69.237338,-69.454406,
        -69.668240,-69.878809,-70.086085,-70.290041,-70.490648,-70.687880,-70.881710,-71.072111,
        -71.259056,-71.442521,-71.622479,-71.798907,-71.971778,-72.141070,-72.306758,-72.468819,
        -72.627230,-72.781969,-72.933013,-73.080341,-73.223931,-73.363763,-73.499816,-73.632070,
        -73.760506,-73.885104,-74.005846,-74.122713,-74.235687,-74.344752,-74.449890,-74.551085,
        -74.648320,-74.741581,-74.830851,-74.916116,-74.997362,-75.074575,-75.147741,-75.216848,
        -75.281884,-75.342835,-75.399692,-75.452441,-75.501074,-75.545580,-75.585949,-75.622172,
        -75.654240,-75.682145,-75.705880,-75.725436,-75.740807,-75.751987,-75.758969,-75.761749,
        -75.760321,-75.754681,-75.744825,-75.730748,-75.712450,-75.689925,-75.663174,-75.632193,
        -75.596982,-75.557540,-75.513867,-75.465963,-75.413829,-75.357466,-75.296876,-75.232061,
        -75.163024,-75.089768,-75.012296,-74.930613,-74.844724,-74.754632,-74.660345,-74.561867,
        -74.459206,-74.352368,-74.241360,-74.126191,-74.006870,-73.883403,-73.755802,-73.624076,
        -73.488235,-73.348290,-73.204252,-73.056132,-72.903943,-72.747698,-72.587408,-72.423088,
        -72.254752,-72.082414,-71.906089,-71.725791,-71.541538,-71.353345,-71.161228,-70.965205,
        -70.765293,-70.561510,-70.353874,-70.142405,-69.927122,-69.708043,-69.485190,-69.258583,
        -69.028243,-68.794190,-68.556448,-68.315038,-68.069983,-67.821305,-67.569029,-67.313178,
        -67.053777,-66.790849,-66.524421,-66.254517,-65.981164,-65.704387,-65.424213,-65.140670,
        -64.853784,-64.563584,-64.270097,-63.973352,-63.673379,-63.370205,-63.063861,-62.754376,
        -62.441782,-62.126108,-61.807386,-61.485647,-61.160922,-60.833244,-60.502645,-60.169158,
        -59.832815,-59.493651,-59.151698,-58.806991,-58.459564,-58.109452,-57.756688,-57.401310,
        -57.043351,-56.682848,-56.319837,-55.954354,-55.586436,-55.216120,-54.843442,-54.468441,
        -54.091153,-53.711618,-53.329872,-52.945955,-52.559905,-52.171761,-51.781562,-51.389348,
        -50.995158,-50.599032,-50.201010,-49.801132,-49.399439,-48.995971,-48.590770,-48.183876,
        -47.775331,-47.365176,-46.953452,-46.540202,-46.125467,-45.709290,-45.291712,-44.872777,
        -44.452526,-44.031003,-43.608251,-43.184311,-42.759229,-42.333046,-41.905806,-41.477553,
        -41.048331,-40.618182,-40.187152,-39.755283,-39.322620,-38.889206,-38.455087,-38.020307,
        -37.584908,-37.148937,-36.712438,-36.275454,-35.838031,-35.400214,-34.962046,-34.523572,
        -34.084838,-33.645888,-33.206766,-32.767519,-32.328189,-31.888822,-31.449464,-31.010158,
        -30.570949,-30.131883,-29.693003,-29.254355,-28.815984,-28.377933,-27.940249,-27.502974,
        -27.066154,-26.629833,-26.194056,-25.758866,-25.324309,-24.890427,-24.457267,-24.024870,
        -23.593282,-23.162546,-22.732706,-22.303806,-21.875888,-21.448997,-21.023175,-20.598466,
        -20.174913,-19.752559,-19.331446,-18.911617,-18.493115,-18.075981,-17.660258,-17.245988,
        -16.833213,-16.421973,-16.012312,-15.604269,-15.197886,-14.793204,-14.390264,-13.989105,
        -13.589769,-13.192294,-12.796722,-12.403092,-12.011443,-11.621814,-11.234244,-10.848772,
        -10.465436,-10.084274,-9.705325,-9.328627,-8.954215,-8.582128,-8.212403,-7.845076,
        -7.480183,-7.117761,-6.757845,-6.400471,-6.045674,-5.693489,-5.343950,-4.997092,
        -4.652949,-4.311554,-3.972940,-3.637141,-3.304190,-2.974118,-2.646958,-2.322742,
        -2.001500,-1.683264,-1.368065,-1.055932,-0.746896,-0.440986,-0.138232,0.161337,
        0.457694,0.750810,1.040656,1.327207,1.610434,1.890312,2.166813,2.439912,
        2.709584,2.975804,3.238546,3.497787,3.753503,4.005670,4.254265,4.499266,
        4.740650,4.978396,5.212482,5.442887,5.669591,5.892573,6.111815,6.327296,
        6.538997,6.746902,6.950991,7.151246,7.347652,7.540191,7.728847,7.913605,
        8.094448,8.271363,8.444335,8.613349,8.778394,8.939454,9.096519,9.249575,
        9.398611,9.543617,9.684580,9.821492,9.954342,10.083121,10.207820,10.328430,
        10.444945,10.557356,10.665656,10.769839,10.869899,10.965831,11.057628,11.145288,
        11.228804,11.308175,11.383397,11.454467,11.521382,11.584142,11.642744,11.697189,
        11.747476,11.793604,11.835575,11.873390,11.907050,11.936558,11.961916,11.983127,
        12.000195,12.013123,12.021916,12.026579,12.027118,12.023538,12.015846,12.004048,
        11.988152,11.968166,11.944098,11.915957,11.883751,11.847491,11.807187,11.762848,
        11.714487,11.662115,11.605744,11.545386,11.481054,11.412761,11.340522,11.264350,
        11.184261,11.100269,11.012390,10.920640,10.825036,10.725594,10.622332,10.515268,
        10.404421,10.289807,10.171448,10.049362,9.923570,9.794091,9.660948,9.524160,
        9.383749,9.239739,9.092151,8.941008,8.786333,8.628151,8.466485,8.301360,
        8.132801,7.960834,7.785483,7.606775,7.424737,7.239396,7.050778,6.858911,
        6.663824,6.465545,6.264102,6.059525,5.851843,5.641086,5.427284,5.210468,
        4.990669,4.767917,4.542245,4.313684,4.082267,3.848025,3.610993,3.371203,
        3.128689,2.883485,2.635624,2.385142,2.132072,1.876451,1.618313,1.357694,
        1.094630,0.829157,0.561312,0.291132,0.018653,-0.256087,-0.533051,-0.812200,
        -1.093497,-1.376904,-1.662381,-1.949891,-2.239393,-2.530850,-2.824220,-3.119465,
        -3.416545,-3.715419,-4.016047,-4.318389,-4.622404,-4.928050,-5.235288,-5.544075,
        -5.854369,-6.166130,-6.479316,-6.793884,-7.109792,-7.426998,-7.745460,-8.065134,
        -8.385979,-8.707950,-9.031006,-9.355103,-9.680198,-10.006247,-10.333208,-10.661035,
        -10.989686,-11.319117,-11.649283,-11.980142,-12.311648,-12.643758,-12.976428,-13.309612,
        -13.643268,-13.977350,-14.311814,-14.646616,-14.981711,-15.317054,-15.652601,-15.988307,
        -16.324128,-16.660019,-16.995935,-17.331831,-17.667664,-18.003388,-18.338958,-18.674330,
        -19.009460,-19.344303,-19.678813,-20.012947,-20.346661,-20.679909,-21.012647,-21.344832,
        -21.676418,-22.007361,-22.337618,-22.667144,-22.995896,-23.323829,-23.650899,-23.977064,
        -24.302279,-24.626500,-24.949685,-25.271791,-25.592773,-25.912589,-26.231196,-26.548552,
        -26.864613,-27.179338,-27.492683,-27.804607,-28.115068,-28.424024,-28.731434,-29.037255,
        -29.341446,-29.643967,-29.944776,-30.243833,-30.541097,-30.836528,-31.130085,-31.421729,
        -31.711420,-31.999118,-32.284785,-32.568381,-32.849867,-33.129205,-33.406357,-33.681284,
        -33.953948,-34.224313,-34.492341,-34.757994,-35.021236,-35.282031,-35.540342,-35.796134,
        -36.049370,-36.300015,-36.548035,-36.793394,-37.036058,-37.275994,-37.513166,-37.747542,
        -37.979088,-38.207771,-38.433560,-38.656421,-38.876323,-39.093234,-39.307122,-39.517958,
        -39.725711,-39.930349,-40.131844,-40.330167,-40.525287,-40.717176,-40.905806,-41.091148,
        -41.273176,-41.451862,-41.627178,-41.799099,-41.967598,-42.132650,-42.294229,-42.452311,
        -42.606870,-42.757883,-42.905325,-43.049175,-43.189408,-43.326002,-43.458935,-43.588186,
        -43.713732,-43.835554,-43.953631,-44.067943,-44.178470,-44.285193,-44.388094,-44.487154,
        -44.582355,-44.673681,-44.761113,-44.844636,-44.924233,-44.999890,-45.071590,-45.139319,
        -45.203063,-45.262808,-45.318540,-45.370248,-45.417917,-45.461537,-45.501096,-45.536582,
        -45.567986,-45.595297,-45.618505,-45.637601,-45.652577,-45.663424,-45.670134,-45.672701,
        -45.671117,-45.665376,-45.655472,-45.641399,-45.623153,-45.600730,-45.574124,-45.543334,
        -45.508355,-45.469185,-45.425822,-45.378265,-45.326512,-45.270562,-45.210416,-45.146074,
        -45.077536,-45.004805,-44.927880,-44.846766,-44.761464,-44.671978,-44.578310,-44.480467,
        -44.378451,-44.272269,-44.161925,-44.047425,-43.928777,-43.805987,-43.679063,-43.548012,
        -43.412843,-43.273565,-43.130186,-42.982718,-42.831170,-42.675552,-42.515877,-42.352155,
        -42.184399,-42.012621,-41.836834,-41.657052,-41.473289,-41.285560,-41.093878,-40.898259,
        -40.698720,-40.495275,-40.287943,-40.076739,-39.861682,-39.642790,-39.420080,-39.193571,
        -38.963284,-38.729237,-38.491451,-38.249946,-38.004743,-37.755864,-37.503330,-37.247164,
        -36.987388,-36.724026,-36.457100,-36.186635,-35.912655,-35.635185,-35.354249,-35.069873,
        -34.782084,-34.490907,-34.196368,-33.898496,-33.597316,-33.292858,-32.985149,-32.674218,
        -32.360093,-32.042805,-31.722381,-31.398854,-31.072252,-30.742606,-30.409948,-30.074309,
        -29.735720,-29.394214,-29.049822,-28.702579,-28.352515,-27.999666,-27.644065,-27.285745,
        -26.924741,-26.561088,-26.194820,-25.825973,-25.454582,-25.080682,-24.704311,-24.325504,
        -23.944297,-23.560729,-23.174835,-22.786654,-22.396223,-22.003580,-21.608763,-21.211811,
        -20.812762,-20.411656,-20.008531,-19.603426,-19.196383,-18.787439,-18.376636,-17.964014,
        -17.549612,-17.133473,-16.715636,-16.296143,-15.875035,-15.452353,-15.028140,-14.602436,
        -14.175284,-13.746726,-13.316805,-12.885562,-12.453040,-12.019283,-11.584332,-11.148232,
        -10.711024,-10.272753,-9.833462,-9.393194,-8.951994,-8.509903,-8.066968,-7.623231,
        -7.178736,-6.733528,-6.287650,-5.841148,-5.394065,-4.946446,-4.498336,-4.049778,
        -3.600817,-3.151499,-2.701867,-2.251966,-1.801842,-1.351538,-0.901100,-0.450572
};
#else // 1/4 smaller table used with interpolation and extra processing.
const float PROGMEM AudioEffectEnsemble::lfoTable[LFO_SAMPLES]={
        0.000000,1.801842,3.600817,5.394065,7.178736,8.951994,10.711024,12.453040,
        14.175284,15.875035,17.549612,19.196383,20.812762,22.396223,23.944297,25.454582,
        26.924741,28.352515,29.735720,31.072252,32.360093,33.597316,34.782084,35.912655,
        36.987388,38.004743,38.963284,39.861682,40.698720,41.473289,42.184399,42.831170,
        43.412843,43.928777,44.378451,44.761464,45.077536,45.326512,45.508355,45.623153,
        45.671117,45.652577,45.567986,45.417917,45.203063,44.924233,44.582355,44.178470,
        43.713732,43.189408,42.606870,41.967598,41.273176,40.525287,39.725711,38.876323,
        37.979088,37.036058,36.049370,35.021236,33.953948,32.849867,31.711420,30.541097,
        29.341446,28.115068,26.864613,25.592773,24.302279,22.995896,21.676418,20.346661,
        19.009460,17.667664,16.324128,14.981711,13.643268,12.311648,10.989686,9.680198,
        8.385979,7.109792,5.854369,4.622404,3.416545,2.239393,1.093497,-0.018653,
        -1.094630,-2.132072,-3.128689,-4.082267,-4.990669,-5.851843,-6.663824,-7.424737,
        -8.132801,-8.786333,-9.383749,-9.923570,-10.404421,-10.825036,-11.184261,-11.481054,
        -11.714487,-11.883751,-11.988152,-12.027118,-12.000195,-11.907050,-11.747476,-11.521382,
        -11.228804,-10.869899,-10.444945,-9.954342,-9.398611,-8.778394,-8.094448,-7.347652,
        -6.538997,-5.669591,-4.740650,-3.753503,-2.709584,-1.610434,-0.457694,0.746896,
        2.001500,3.304190,4.652949,6.045674,7.480183,8.954215,10.465436,12.011443,
        13.589769,15.197886,16.833213,18.493115,20.174913,21.875888,23.593282,25.324309,
        27.066154,28.815984,30.570949,32.328189,34.084838,35.838031,37.584908,39.322620,
        41.048331,42.759229,44.452526,46.125467,47.775331,49.399439,50.995158,52.559905,
        54.091153,55.586436,57.043351,58.459564,59.832815,61.160922,62.441782,63.673379,
        64.853784,65.981164,67.053777,68.069983,69.028243,69.927122,70.765293,71.541538,
        72.254752,72.903943,73.488235,74.006870,74.459206,74.844724,75.163024,75.413829,
        75.596982,75.712450,75.760321,75.740807,75.654240,75.501074,75.281884,74.997362,
        74.648320,74.235687,73.760506,73.223931,72.627230,71.971778,71.259056,70.490648,
        69.668240,68.793612,67.868641,66.895294,65.875623,64.811767,63.705940,62.560435,
        61.377614,60.159906,58.909802,57.629852,56.322659,54.990872,53.637187,52.264335,
        50.875084,49.472228,48.058586,46.636994,45.210301,43.781365,42.353047,40.928203,
        39.509683,38.100324,36.702945,35.320339,33.955275,32.610484,31.288662,29.992459,
        28.724477,27.487268,26.283321,25.115068,23.984871,22.895022,21.847737,20.845153,
        19.889325,18.982221,18.125715,17.321592,16.571536,15.877132,15.239863,14.661104,
        14.142122,13.684074,13.288005,12.954841,12.685397,12.480367,12.340326,12.265728,
        12.256907,12.314076,12.437324,12.626618,12.881804,13.202605,13.588623,14.039339,
        14.554113,15.132189,15.772691,16.474629,17.236899,18.058284,18.937460,19.872992,
        20.863343,21.906873,23.001845,24.146424,25.338685,26.576612,27.858105,29.180983,
        30.542987,31.941786,33.374979,34.840100,36.334625,37.855973,39.401512,40.968566,
        42.554416,44.156307,45.771455,47.397049,49.030256,50.668228,52.308108,53.947033,
        55.582139,57.210567,58.829472,60.436019,62.027400,63.600828,65.153550,66.682847,
        68.186042,69.660505,71.103656,72.512969,73.885980,75.220289,76.513563,77.763547,
        78.968057,80.124996,81.232349,82.288190,83.290685,84.238099,85.128791,85.961226,
        86.733970,87.445700,88.095199,88.681364,89.203206,89.659851,90.050543,90.374644,
        90.631636,90.821122,90.942826,90.996595,90.982397,90.900324,90.750589,90.533527,
        90.249594,89.899366,89.483541,89.002931,88.458468,87.851195,87.182273,86.452969,
        85.664662,84.818834,83.917073,82.961065,81.952596,80.893543,79.785876,78.631650,
        77.433005,76.192160,74.911408,73.593114,72.239709,70.853689,69.437604,67.994060,
        66.525710,65.035250,63.525416,61.998976,60.458729,58.907494,57.348111,55.783432,
        54.216317,52.649628,51.086226,49.528963,47.980677,46.444190,44.922299,43.417774,
        41.933350,40.471723,39.035548,37.627430,36.249921,34.905515,33.596645,32.325675,
        31.094901,29.906542,28.762737,27.665543,26.616930,25.618777,24.672869,23.780894,
        22.944439,22.164990,21.443925,20.782513,20.181914,19.643173,19.167224,18.754880,
        18.406839,18.123678,17.905856,17.753708,17.667448,17.647170,17.692843,17.804316,
        17.981313,18.223440,18.530180,18.900897,19.334835,19.831124,20.388774,21.006684,
        21.683641,22.418322,23.209296,24.055028,24.953881,25.904120,26.903913,27.951337,
        29.044380,30.180943,31.358850,32.575842,33.829593,35.117702,36.437708,37.787089,
        39.163266,40.563611,41.985449,43.426064,44.882705,46.352590,47.832909,49.320833,
        50.813518,52.308108,53.801743,55.291561,56.774707,58.248337,59.709622,61.155752,
        62.583945,63.991451,65.375553,66.733578,68.062897,69.360933,70.625163,71.853125,
        73.042422,74.190725,75.295779,76.355405,77.367508,78.330074,79.241182,80.099000,
        80.901795,81.647929,82.335869,82.964185,83.531555,84.036766,84.478717,84.856422,
        85.169010,85.415726,85.595936,85.709125,85.754897,85.732980,85.643223,85.485597,
        85.260196,84.967236,84.607055,84.180111,83.686984,83.128372,82.505092,81.818077,
        81.068374,80.257146,79.385662,78.455304,77.467557,76.424011,75.326356,74.176379,
        72.975959,71.727071,70.431771,69.092201,67.710584,66.289214,64.830460,63.336757,
        61.810599,60.254543,58.671195,57.063211,55.433290,53.784171,52.118624,50.439449,
        48.749471,47.051530,45.348480,43.643186,41.938511,40.237318,38.542462,36.856785,
        35.183109,33.524236,31.882936,30.261948,28.663971,27.091661,25.547626,24.034421,
        22.554542,21.110425,19.704437,18.338874,17.015959,15.737835,14.506559,13.324104,
        12.192352,11.113091,10.088010,9.118701,8.206649,7.353235,6.559733,5.827303,
        5.156994,4.549741,4.006361,3.527552,3.113897,2.765853,2.483760,2.267835,
        2.118173,2.034745,2.017403,2.065875,2.179767,2.358564,2.601634,2.908224,
        3.277463,3.708367,4.199836,4.750658,5.359514,6.024974,6.745506,7.519476,
        8.345149,9.220699,10.144202,11.113648,12.126944,13.181912,14.276299,15.407777,
        16.573951,17.772361,19.000486,20.255751,21.535528,22.837147,24.157893,25.495016,
        26.845738,28.207250,29.576726,30.951324,32.328189,33.704464,35.077289,36.443812,
        37.801191,39.146597,40.477224,41.790293,43.083054,44.352793,45.596839,46.812565,
        47.997396,49.148812,50.264353,51.341622,52.378294,53.372116,54.320910,55.222583,
        56.075125,56.876615,57.625223,58.319219,58.956967,59.536935,60.057695,60.517928,
        60.916423,61.252080,61.523916,61.731060,61.872761,61.948386,61.957421,61.899474,
        61.774274,61.581671,61.321639,60.994274,60.599793,60.138535,59.610962,59.017655,
        58.359314,57.636757,56.850919,56.002850,55.093713,54.124780,53.097435,52.013164,
        50.873559,49.680312,48.435210,47.140138,45.797067,44.408060,42.975258,41.500885,
        39.987240,38.436690,36.851673,35.234686,33.588285,31.915080,30.217728,28.498929,
        26.761424,25.007985,23.241414,21.464536,19.680193,17.891243,16.100549,14.310977,
        12.525393,10.746651,8.977596,7.221053,5.479822,3.756677,2.054358,0.375565,
        -1.277044,-2.900860,-4.493326,-6.051944,-7.574279,-9.057961,-10.500691,-11.900246,
        -13.254484,-14.561342,-15.818849,-17.025120,-18.178367,-19.276898,-20.319122,-21.303550,
        -22.228800,-23.093600,-23.896784,-24.637305,-25.314225,-25.926728,-26.474111,-26.955794,
        -27.371315,-27.720335,-28.002635,-28.218119,-28.366815,-28.448869,-28.464554,-28.414260,
        -28.298502,-28.117911,-27.873239,-27.565355,-27.195243,-26.764002,-26.272841,-25.723081,
        -25.116148,-24.453574,-23.736992,-22.968136,-22.148832,-21.281001,-20.366652,-19.407879,
        -18.406857,-17.365840,-16.287154,-15.173192,-14.026414,-12.849341,-11.644545,-10.414655,
        -9.162339,-7.890312,-6.601321,-5.298146,-3.983593,-2.660488,-1.331673,-0.000000,
        1.331673,2.660488,3.983593,5.298146,6.601321,7.890312,9.162339,10.414655,
        11.644545,12.849341,14.026414,15.173192,16.287154,17.365840,18.406857,19.407879,
        20.366652,21.281001,22.148832,22.968136,23.736992,24.453574,25.116148,25.723081,
        26.272841,26.764002,27.195243,27.565355,27.873239,28.117911,28.298502,28.414260,
        28.464554,28.448869,28.366815,28.218119,28.002635,27.720335,27.371315,26.955794,
        26.474111,25.926728,25.314225,24.637305,23.896784,23.093600,22.228800,21.303550,
        20.319122,19.276898,18.178367,17.025120,15.818849,14.561342,13.254484,11.900246,
        10.500691,9.057961,7.574279,6.051944,4.493326,2.900860,1.277044,-0.375565,
        -2.054358,-3.756677,-5.479822,-7.221053,-8.977596,-10.746651,-12.525393,-14.310977,
        -16.100549,-17.891243,-19.680193,-21.464536,-23.241414,-25.007985,-26.761424,-28.498929,
        -30.217728,-31.915080,-33.588285,-35.234686,-36.851673,-38.436690,-39.987240,-41.500885,
        -42.975258,-44.408060,-45.797067,-47.140138,-48.435210,-49.680312,-50.873559,-52.013164,
        -53.097435,-54.124780,-55.093713,-56.002850,-56.850919,-57.636757,-58.359314,-59.017655,
        -59.610962,-60.138535,-60.599793,-60.994274,-61.321639,-61.581671,-61.774274,-61.899474,
        -61.957421,-61.948386,-61.872761,-61.731060,-61.523916,-61.252080,-60.916423,-60.517928,
        -60.057695,-59.536935,-58.956967,-58.319219,-57.625223,-56.876615,-56.075125,-55.222583,
        -54.320910,-53.372116,-52.378294,-51.341622,-50.264353,-49.148812,-47.997396,-46.812565,
        -45.596839,-44.352793,-43.083054,-41.790293,-40.477224,-39.146597,-37.801191,-36.443812,
        -35.077289,-33.704464,-32.328189,-30.951324,-29.576726,-28.207250,-26.845738,-25.495016,
        -24.157893,-22.837147,-21.535528,-20.255751,-19.000486,-17.772361,-16.573951,-15.407777,
        -14.276299,-13.181912,-12.126944,-11.113648,-10.144202,-9.220699,-8.345149,-7.519476,
        -6.745506,-6.024974,-5.359514,-4.750658,-4.199836,-3.708367,-3.277463,-2.908224,
        -2.601634,-2.358564,-2.179767,-2.065875,-2.017403,-2.034745,-2.118173,-2.267835,
        -2.483760,-2.765853,-3.113897,-3.527552,-4.006361,-4.549741,-5.156994,-5.827303,
        -6.559733,-7.353235,-8.206649,-9.118701,-10.088010,-11.113091,-12.192352,-13.324104,
        -14.506559,-15.737835,-17.015959,-18.338874,-19.704437,-21.110425,-22.554542,-24.034421,
        -25.547626,-27.091661,-28.663971,-30.261948,-31.882936,-33.524236,-35.183109,-36.856785,
        -38.542462,-40.237318,-41.938511,-43.643186,-45.348480,-47.051530,-48.749471,-50.439449,
        -52.118624,-53.784171,-55.433290,-57.063211,-58.671195,-60.254543,-61.810599,-63.336757,
        -64.830460,-66.289214,-67.710584,-69.092201,-70.431771,-71.727071,-72.975959,-74.176379,
        -75.326356,-76.424011,-77.467557,-78.455304,-79.385662,-80.257146,-81.068374,-81.818077,
        -82.505092,-83.128372,-83.686984,-84.180111,-84.607055,-84.967236,-85.260196,-85.485597,
        -85.643223,-85.732980,-85.754897,-85.709125,-85.595936,-85.415726,-85.169010,-84.856422,
        -84.478717,-84.036766,-83.531555,-82.964185,-82.335869,-81.647929,-80.901795,-80.099000,
        -79.241182,-78.330074,-77.367508,-76.355405,-75.295779,-74.190725,-73.042422,-71.853125,
        -70.625163,-69.360933,-68.062897,-66.733578,-65.375553,-63.991451,-62.583945,-61.155752,
        -59.709622,-58.248337,-56.774707,-55.291561,-53.801743,-52.308108,-50.813518,-49.320833,
        -47.832909,-46.352590,-44.882705,-43.426064,-41.985449,-40.563611,-39.163266,-37.787089,
        -36.437708,-35.117702,-33.829593,-32.575842,-31.358850,-30.180943,-29.044380,-27.951337,
        -26.903913,-25.904120,-24.953881,-24.055028,-23.209296,-22.418322,-21.683641,-21.006684,
        -20.388774,-19.831124,-19.334835,-18.900897,-18.530180,-18.223440,-17.981313,-17.804316,
        -17.692843,-17.647170,-17.667448,-17.753708,-17.905856,-18.123678,-18.406839,-18.754880,
        -19.167224,-19.643173,-20.181914,-20.782513,-21.443925,-22.164990,-22.944439,-23.780894,
        -24.672869,-25.618777,-26.616930,-27.665543,-28.762737,-29.906542,-31.094901,-32.325675,
        -33.596645,-34.905515,-36.249921,-37.627430,-39.035548,-40.471723,-41.933350,-43.417774,
        -44.922299,-46.444190,-47.980677,-49.528963,-51.086226,-52.649628,-54.216317,-55.783432,
        -57.348111,-58.907494,-60.458729,-61.998976,-63.525416,-65.035250,-66.525710,-67.994060,
        -69.437604,-70.853689,-72.239709,-73.593114,-74.911408,-76.192160,-77.433005,-78.631650,
        -79.785876,-80.893543,-81.952596,-82.961065,-83.917073,-84.818834,-85.664662,-86.452969,
        -87.182273,-87.851195,-88.458468,-89.002931,-89.483541,-89.899366,-90.249594,-90.533527,
        -90.750589,-90.900324,-90.982397,-90.996595,-90.942826,-90.821122,-90.631636,-90.374644,
        -90.050543,-89.659851,-89.203206,-88.681364,-88.095199,-87.445700,-86.733970,-85.961226,
        -85.128791,-84.238099,-83.290685,-82.288190,-81.232349,-80.124996,-78.968057,-77.763547,
        -76.513563,-75.220289,-73.885980,-72.512969,-71.103656,-69.660505,-68.186042,-66.682847,
        -65.153550,-63.600828,-62.027400,-60.436019,-58.829472,-57.210567,-55.582139,-53.947033,
        -52.308108,-50.668228,-49.030256,-47.397049,-45.771455,-44.156307,-42.554416,-40.968566,
        -39.401512,-37.855973,-36.334625,-34.840100,-33.374979,-31.941786,-30.542987,-29.180983,
        -27.858105,-26.576612,-25.338685,-24.146424,-23.001845,-21.906873,-20.863343,-19.872992,
        -18.937460,-18.058284,-17.236899,-16.474629,-15.772691,-15.132189,-14.554113,-14.039339,
        -13.588623,-13.202605,-12.881804,-12.626618,-12.437324,-12.314076,-12.256907,-12.265728,
        -12.340326,-12.480367,-12.685397,-12.954841,-13.288005,-13.684074,-14.142122,-14.661104,
        -15.239863,-15.877132,-16.571536,-17.321592,-18.125715,-18.982221,-19.889325,-20.845153,
        -21.847737,-22.895022,-23.984871,-25.115068,-26.283321,-27.487268,-28.724477,-29.992459,
        -31.288662,-32.610484,-33.955275,-35.320339,-36.702945,-38.100324,-39.509683,-40.928203,
        -42.353047,-43.781365,-45.210301,-46.636994,-48.058586,-49.472228,-50.875084,-52.264335,
        -53.637187,-54.990872,-56.322659,-57.629852,-58.909802,-60.159906,-61.377614,-62.560435,
        -63.705940,-64.811767,-65.875623,-66.895294,-67.868641,-68.793612,-69.668240,-70.490648,
        -71.259056,-71.971778,-72.627230,-73.223931,-73.760506,-74.235687,-74.648320,-74.997362,
        -75.281884,-75.501074,-75.654240,-75.740807,-75.760321,-75.712450,-75.596982,-75.413829,
        -75.163024,-74.844724,-74.459206,-74.006870,-73.488235,-72.903943,-72.254752,-71.541538,
        -70.765293,-69.927122,-69.028243,-68.069983,-67.053777,-65.981164,-64.853784,-63.673379,
        -62.441782,-61.160922,-59.832815,-58.459564,-57.043351,-55.586436,-54.091153,-52.559905,
        -50.995158,-49.399439,-47.775331,-46.125467,-44.452526,-42.759229,-41.048331,-39.322620,
        -37.584908,-35.838031,-34.084838,-32.328189,-30.570949,-28.815984,-27.066154,-25.324309,
        -23.593282,-21.875888,-20.174913,-18.493115,-16.833213,-15.197886,-13.589769,-12.011443,
        -10.465436,-8.954215,-7.480183,-6.045674,-4.652949,-3.304190,-2.001500,-0.746896,
        0.457694,1.610434,2.709584,3.753503,4.740650,5.669591,6.538997,7.347652,
        8.094448,8.778394,9.398611,9.954342,10.444945,10.869899,11.228804,11.521382,
        11.747476,11.907050,12.000195,12.027118,11.988152,11.883751,11.714487,11.481054,
        11.184261,10.825036,10.404421,9.923570,9.383749,8.786333,8.132801,7.424737,
        6.663824,5.851843,4.990669,4.082267,3.128689,2.132072,1.094630,0.018653,
        -1.093497,-2.239393,-3.416545,-4.622404,-5.854369,-7.109792,-8.385979,-9.680198,
        -10.989686,-12.311648,-13.643268,-14.981711,-16.324128,-17.667664,-19.009460,-20.346661,
        -21.676418,-22.995896,-24.302279,-25.592773,-26.864613,-28.115068,-29.341446,-30.541097,
        -31.711420,-32.849867,-33.953948,-35.021236,-36.049370,-37.036058,-37.979088,-38.876323,
        -39.725711,-40.525287,-41.273176,-41.967598,-42.606870,-43.189408,-43.713732,-44.178470,
        -44.582355,-44.924233,-45.203063,-45.417917,-45.567986,-45.652577,-45.671117,-45.623153,
        -45.508355,-45.326512,-45.077536,-44.761464,-44.378451,-43.928777,-43.412843,-42.831170,
        -42.184399,-41.473289,-40.698720,-39.861682,-38.963284,-38.004743,-36.987388,-35.912655,
        -34.782084,-33.597316,-32.360093,-31.072252,-29.735720,-28.352515,-26.924741,-25.454582,
        -23.944297,-22.396223,-20.812762,-19.196383,-17.549612,-15.875035,-14.175284,-12.453040,
        -10.711024,-8.951994,-7.178736,-5.394065,-3.600817,-1.801842
};
#endif
